/*
 *  polyFactor.cpp
 *  fp
 *
 *  Created by Robert Delaney on 6/4/13.
 *  Copyright 2013 Bob Delaney's Science Software. All rights reserved.
 *
 */

#include "fp.h"
#include "poly.h"
#include "polyMath.h"
#include "matq.h"
#include "matqMath.h"
#include "matqConv.h"
#include "polyFactor.h"
#include "polyConv.h"


// Yun's algorithm
// http://en.wikipedia.org/wiki/Square-free_polynomial
// if x = a1 * a2^2 * a3^3 * ... * an^n
// then  row 0 of z contains a2 * a3^2 * a4^3 * ... * an^(n-1)
// row 1 contains a1, row 2 contains a2, ... , row n contains an
void polySquareFree(poly*& z, int& numPolys, const poly& x)
{
    poly        a, b, c, d;
    INT32       i;
    bf          temp;
    poly        pd;
    
    numPolys = 1;
    // x.deg is max number of factors
    z = (poly*)malloc(x.deg*sizeof(poly));
    for(i=0;i<x.deg;++i)
        init(z[i]);
    
    if(x.deg<2)
    {
        z[0] =  x;
        return;
    }
    
    pd = polyDeriv(x);
    a = gcd(x, pd); // a0  need to makeMonic
    //cout << "a = " << a << endl; // -2, 2 for x^2 - 2x + 1
    z[numPolys-1] = a;
    b = x / a;
    c = pd / a;
    d = c - polyDeriv(b);
    while(!(b==1))
    {
        //cout << "top b = " << b << endl;
        a = gcd(b, d); // a1, a2, ...
        if(a!=1)
        {
            numPolys++;
            z[numPolys-1] = a;
        }
        b = b / a;
        c = d / a;
        d = c - polyDeriv(b);
       // cout << "bottom b = " << b << endl << endl;
    }
    
    for(i=0;i<numPolys;++i)
        z[i] = polyOverZ(z[i]);
    
}/* polySquareFree */

// Yun's algorithm
// http://en.wikipedia.org/wiki/Square-free_polynomial
// if x = a1 * a2^2 * a3^3 * ... * an^n
// then  row 0 of z contains a0 = a2 * a3^2 * a4^3 * ... * an^(n-1)
// row 1 contains a1, row 2 contains a2, ... , row n contains an
void polySquareFreeModq(poly*& z, int& numPolys, const poly& x, double q)
{
    poly        a, b, c, d, e, xt, r;
    INT32       i;
    bf          temp;
    
    xt = x % q;
    xt = polyNormalize(xt);
    
    numPolys = 1;
    // xt.deg is max number of factors
    z = (poly*)malloc(xt.deg*sizeof(poly));
    for(i=0;i<x.deg;++i)
        init(z[i]);
    
    if(xt.deg<2)
    {
        z[0] =  xt;
        return;
    }
    
    // f = xt; c = a; g = e; w = b
    e = polyDeriv(xt);
    makeMonic(e, e, q);
   // e = polyReduce(e);
    a = gcd(xt, e, q); // a0
    z[numPolys-1] = a;
    //b = xt / a;
    divEuclid(b, r, xt, a, q);
    //makeMonic(b, b, q);
    //c = polyDeriv(xt) / a;
    divEuclid(c, r, e, a, q);
    //makeMonic(c, c, q);
    e = polyDeriv(b);
    //makeMonic(e, e, q);
    d = (c - e) % q;  // d=0 is OK
    d = polyReduce(d);
    while(!(b==1))
    {
        a = gcd(b, d, q); // a1, a2, ...
        if(a!=1)
        {
            numPolys++;
            z[numPolys-1] = a;
        }
        //b = b / a;
        divEuclid(b, r, b, a, q);
        //makeMonic(b, b, q);
        //c = d / a;
        divEuclid(c, r, d, a, q);
       // makeMonic(c, c, q);
        e = polyDeriv(b);
        //makeMonic(e, e, q);
        d = (c - e) % q;
        //makeMonic(d, d, q);
    }
    
    for(i=0;i<numPolys;++i)
        z[i] = polyOverZ(z[i]);
    
}/* polySquareFreeModq */


void findPowersOfSquareFreeFactors(char*& zString, int& numPolys, const poly& x)
{
    poly        *z;
    poly        zt, r, q, temp;
    int        i;
    int        *theExp;
    bool        isGood;
    int        length;
    char        *xString, *numString;
    
    polySquareFree(z, numPolys, x);
    
    if(numPolys<1)
        return;
    
    if(numPolys==1)
    {
        polyConvToString(zString, z[0]);
        return;
    }
    
    numString = (char*)malloc(10*sizeof(char));
    
    theExp = (int*)malloc(numPolys*sizeof(int));
    for(i=0;i<numPolys;i++)
        theExp[i] = 0;
    
    
    //zt contains powers of some or all of z[1], z[2], ...
    for(i=1;i<numPolys;i++)
    {
        zt = z[0];
        isGood = true;
        while(isGood && z[i].deg)
        {
            divEuclid(q, r, zt, z[i]);
            if(!r.deg && !r.array[0])
            {
                theExp[i]++;
                zt = q;
            }
            else
            {
                isGood = false;
            }
        }
    }
    
    length = 0;
    for(i=1;i<numPolys;i++)
    {
        polyConvToString(xString, z[i]);
        length = length + strlen(xString);
        free(xString);
    }
    length+=(8*numPolys);
    
    zString = (char*)malloc(length*sizeof(char));
    
    zString[0] = 0;
    
    for(i=1;i<numPolys;i++)
    {
        polyConvToString(xString, z[i]);
        strcat(zString, "(");
        strcat(zString, xString);
        strcat(zString, ")");
        if(theExp[i]>0)
        {
            strcat(zString, "^");
            sprintf(numString,"%d", theExp[i]+1);
            strcat(zString, numString);
        }
        if(i!=(numPolys-1))
            strcat(zString, " ");
    }
    
    numPolys--;
    
    free(theExp);
    free(numString);
    
}/* findPowersOfSquareFreeFactors */


void findPowersOfSquareFreeFactors(poly*& z1, int*& theExp, int& numPolys, const poly& x)
{
    poly        *z;
    poly        zt, r, q, temp;
    int         i;
    bool        isGood;
    
    polySquareFree(z, numPolys, x);
    
    if(numPolys<=1)
    {
        z1 = (poly*)malloc(sizeof(poly));
        init(z1[0], x.deg);
        theExp = (int*)malloc(sizeof(int));
        z1[0] = x;
        theExp[0] = 1;
        return;
    }
    
    theExp = (int*)malloc((numPolys-1)*sizeof(int));
    for(i=0;i<(numPolys-1);i++)
        theExp[i] = 0;
    
    for(i=1;i<numPolys;i++)
    {
        zt = z[0]; //zt contains powers of some or all of z[1], z[2], ...
        isGood = true;
        while(isGood && z[i].deg)
        {
            divEuclid(q, r, zt, z[i]);
            if(!r.deg && !r.array[0])
            {
                theExp[i-1]++;
                zt = q;
            }
            else
            {
                isGood = false;
            }
        }
    }
    
    z1 = (poly*)malloc((numPolys-1)*sizeof(poly));
    
    for(i=0;i<(numPolys-1);i++)
    {
        init(z1[i]);
        z1[i] = z[i+1];
        theExp[i]++;
    }
    
    numPolys--;
    
    //free(z);
    
}/* findPowersOfSquareFreeFactors */


static void polyFactorFirst(matq& factors, matq& multiplicity, poly& a, double& q, const poly& thePoly)
{
    INT32       i, numPolys;
    poly        x;
    mb          tempmb;
    bf          tempbf;
    matq        Q, z, theExpColM;
    matq        reduced, nullVectors;
    
    
    if(thePoly.array==NULL || thePoly.deg<0)
    {
        init(factors, 1, 1);
        factors.array[0][0] = 0;
        init(multiplicity, 1, 1);
        multiplicity.array[0][0] = 0;
        init(a, 0);
        a.array[0] = 1;
        return;
    }
    
    if(thePoly.deg<2)
    {
       if(thePoly.deg==0)
       {
           init(factors, 1, 1);
           factors.array[0][0] = thePoly.array[0];
           init(multiplicity, 1, 1);
           multiplicity.array[0][0] = 1;
           init(a, 0);
           a.array[0] = 1;
           return;
       }
        // now thePoly.deg==1
        init(factors, 1, 2);
        factors.array[0][0] = thePoly.array[0];
        factors.array[0][1] = thePoly.array[1];
        init(multiplicity, 1, 1);
        multiplicity.array[0][0] = 1;
        a = thePoly;
        return;
    }
    
    // now thePoly.deg > 1
    // Check coeff. nums. for common integer factor and divide that out.
    x = thePoly;  // so can modify x
    tempmb = 1;
    for(i=0;i<=x.deg;++i)
        tempmb = gcd(tempmb, abs(x.array[i].num));
    if(tempmb>1)
    {
        tempbf = tempmb;
        x = x / tempbf;
    }
    
    
    //  Check for actual fractions and if so convert x to integer poly.
    
    // get lcd of poly
    tempmb = lcd(x);
    if(tempmb>1)
    {
        tempbf = tempmb;
        x = x * tempbf;
    }
    // Pass the matrix as x to remove square and above factors
    // findPowersOfSquareFreeFactors(matq& z, matq& theExpColM, int& numPolys, const poly& x);
    findPowersOfSquareFreeFactors(z, theExpColM, numPolys, x);
    
    // if a1, now  called a, is of degree one we are done
    init(a, z.nc-1);
    for(i=0;i<=a.deg;++i)
        a.array[i] = z.array[0][i];
    
    a = polyNormalize(a);
    if(a.deg<2)
    {
        factors = z;
        multiplicity = theExpColM;
        return;
    }
    
    //for now choose q as the smallest prime greater than 2 x the max coeff. of a in abs value
    tempmb = 1;
    for(i=0;i<=a.deg;++i)
    {
        if(abs(a.array[i].num)>tempmb)
            tempmb = abs(a.array[i].num);
    }
    
    tempmb = tempmb*2 + 1;
    while(!isItPrime(tempmb, 35))
    {
        tempmb+=2;
    }
    equate(q, tempmb);

   
}/* polyFactorFirst */

bool polyFactorOverIntegers(poly*& factors, int& numFactors, matq& multiplicity, const poly& thePoly)
{
    poly   a, *polyNV;
    matq    Q, matqFactors, nullVectors;
    matq    multiplicitySF; // for square free
    double  q;
    int     i, numNV, numPolys;
    poly    *factorsNV;
    int     *theExp;
    poly    *sfFactors;
    mb      tempmb;
    
    //polyFactorFirst(matqFactors, multiplicitySF, a, q, thePoly);
    // void findPowersOfSquareFreeFactors(poly*& z1, int*& theExp, int& numPolys, const poly& x);
   // q=11;
    
    findPowersOfSquareFreeFactors(sfFactors, theExp, numPolys, thePoly );
    if(numPolys<1)
        return false;
    
    a = sfFactors[0];
    if(a.array[a.deg]!=1)
        return false;
    
    //for now choose q as the smallest prime greater than 2 x the max coeff. of a in abs value
    tempmb = 1;
    for(i=0;i<=a.deg;++i)
    {
        if(abs(a.array[i].num)>tempmb)
            tempmb = abs(a.array[i].num);
    }
    
    tempmb = tempmb*2 + 1;
    while(!isItPrime(tempmb, 35))
    {
        tempmb+=2;
    }
    equate(q, tempmb);
    //q = 2;
    numFactors = 0;
    
    polyCalcQmatrix(Q, a, q);
    Q = Q - matqUnit(Q.nr);
    matqNullSpaceModq(numNV, nullVectors, Q, q); // if a is irreducible then numNV=0 should happen
    if(!numNV)
    {
        // a is irreducible, so change nullvectors column to contain a
        numNV = 1;
        //myFree(factorsNV, 1);
        factorsNV = (poly*)malloc(sizeof(poly));
        init(factorsNV[0], a.deg);
        for(i=0;i<=a.deg;++i)
            factorsNV[0].array[i] = a.array[i];
        
        goto finish;
    }
    convNullVectorsToPolys(polyNV, nullVectors, q);
    if(!polyFindAllFactors(factorsNV, a, polyNV, numFactors, numNV, q))
    {
        // a is irreducible, so change nullvectors column to contain a
        numFactors = numNV = 1;
        //myFree(factorsNV, 1);
        factorsNV = (poly*)malloc(sizeof(poly));
        init(factorsNV[0], a.deg);
        for(i=0;i<=a.deg;++i)
            factorsNV[0].array[i] = a.array[i];
    }
    
finish:
    // now combine factors, their number is matqFactors.nc + numNV
    numFactors = numNV + numPolys - 1; // we skip one in square free
    factors = (poly*)malloc(numFactors*sizeof(poly));
    for(i=0;i<numFactors;i++)
        init(factors[i]);
    
    init(multiplicity, numFactors, 1);
    
    // first load in the factors from the null vectors and multiplicity = 1
    for(i=0;i<numNV;++i)
    {
        factors[i] = factorsNV[i];
        multiplicity.array[i][0] = 1;
    }
    
    // next load in square free factoes, skipping sfFactors[0] which is a
    for(i=0;i<numPolys-1;++i)
    {
        factors[numNV+i] = sfFactors[i+1];
        multiplicity.array[numNV+i][0] = theExp[i+1];
    }
    
    return true;
    
}/* polyFactorOverIntegers */

// http://www.science.unitn.it/~degraaf/compalg/polfact.pdf
static void polyFactorOverpActual(poly*& factors, int& numFactors, const poly& f, double p)
{
    int         i, numNV;
    matq        Q, nullVectors;
    poly        *polyNV;
    
    //cout  << f << endl;
    polyCalcQmatrix(Q, f, p);
    Q = Q - matqUnit(Q.nr);
    matqNullSpaceModq(numNV, nullVectors, Q, p); // if f is irreducible then numFactors=0 should happen
    if(!numNV)
    {
        // f is irreducible, so change nullvectors column to contain f
        numFactors = 1;
        init(factors[0], f.deg);
        for(i=0;i<=f.deg;++i)
            factors[0].array[i] = f.array[i];
        return;
    }
    
    convNullVectorsToPolys(polyNV, nullVectors, p);
    
    if(!polyFindAllFactors(factors, f, polyNV, numFactors, numNV,  p))
    {
        // f is irreducible, so change nullvectors column to contain f
        numFactors = 1;
        init(factors[0], f.deg);
        for(i=0;i<=f.deg;++i)
            factors[0].array[i] = f.array[i];
        return;
    }
    
}/* polyFactorOverpActual */

// http://www.science.unitn.it/~degraaf/compalg/polfact.pdf
// give this poly f = g^e , it returns e and g, and true if fDeriv==0 false if not
static bool findPolyPower(poly& g, int& e, const poly& f, double q)
{
    poly    fDeriv, h, qt, r, ft, gt, d;
    int     i, qInt;
    
    if(f.deg==1 || !f.deg)
    {
        // cannot have power>1
        gt = f;
        g = gt;
        e = 1;
        return false;
    }
    
    fDeriv = polyDeriv(f) % q;
    fDeriv = polyNormalize(fDeriv);
    makeMonic(d, fDeriv, q);
    
    if(d==0)
    {
        // f = h(x^q) = h(x)^q , so replace x^q by x
        qInt = q;
        init(h, f.deg/qInt);
        for(i=0;i<=h.deg;++i)
            h.array[i] = f.array[qInt*i];
        g = h;
        e = qInt;
        return true;
    }
    // now fDeriv is not zero
    r = gcd(f, d, q);
    divEuclid(gt, r, f, r, q);
    // now f = gt^e , so successively divide f by g until get one
    ft = f;
    e = 0;
    while(ft != 1)
    {
        divEuclid(ft, r, ft, gt, q);
        e++;
    }
    g = gt;
    return false;
    
}/* findPolyPower */

// http://www.science.unitn.it/~degraaf/compalg/polfact.pdf
static void polyFactorOverpFirst(poly*& factors, int& numFactors, int* power, const poly& f, double p)
{
    int     i, e, et;
    poly    temp;
    bool    again;
    
    polyFactorOverpActual(factors, numFactors, f, p);
    
    /*
    if(numFactors==1 && factors[0]==f)
    {
        power[0] = 1;
        return;
    }
     */
    //cout << f << endl;
    //for(i=0;i<numFactors;++i)
    //    cout << factors[i] << endl;
    
    for(i=0;i<numFactors;++i)
    {
        e = 1;
        temp = factors[i];
        if(temp.deg>1)
        {
            do
            {
                again = findPolyPower(temp, et, temp, p);
                e = e * et;
            }
            while (again);
            
            myFree(factors[i]);
            factors[i] = temp;
        }
        power[i] = e;
    }
    
}/* polyFactorOverpFirst */

void polyFactorOverpPenUlt(poly*& factors, int& numFactors, int*& power, const poly& f, double p)
{
    poly    *factors1, *factors2;
    poly    ft;
    int     *power1, *power2;
    int     numFactors1, numFactors2;
    int     i, i1, i2, i3;
    
    if(f.array[f.deg]!=1)
    {
        numFactors = 0;
        return;
    }
    
    if(f.deg<2)
    {
        numFactors = 1;
        factors = (poly*)malloc(sizeof(poly));
        init(factors[0], f.deg);
        for(i=0;i<=f.deg;++i)
            factors[0].array[i] = f.array[i];
        power = (int*)malloc(sizeof(int));
        power[0] = 1;
        return;
    }
    
    factors1 = (poly*)malloc(f.deg*sizeof(poly));
    for(i=0;i<f.deg;++i)
        init(factors1[i]);
    power1 = (int*)malloc(f.deg*sizeof(int));
    
    polyFactorOverpFirst(factors1, numFactors1, power1, f, p);
    
    factors = (poly*)malloc(f.deg*sizeof(poly));
    for(i=0;i<f.deg;++i)
        init(factors[i]);
    power = (int*)malloc(f.deg*sizeof(int));
    
    factors2 = (poly*)malloc(f.deg*sizeof(poly));
    for(i=0;i<f.deg;++i)
        init(factors2[i]);
    
    power2 = (int*)malloc(f.deg*sizeof(int));
    
    i3 = 0;
    numFactors = 0;
    for(i1=0;i1<numFactors1;++i1)
    {
        ft = factors1[i1];
        if(!ft.deg)
            continue;
        
        if(ft.deg==1)
        {
            numFactors+=1;
            factors[i3] = ft;
            power[i3++] = power1[i1];
        }
        else
        {
            polyFactorOverpActual(factors2, numFactors2, ft, p);
            numFactors+=numFactors2;
            for(i2=0;i2<numFactors2;++i2)
            {
                factors[i3] = factors2[i2];
                power[i3++] = power1[i1];
            }
            for(i2=0;i2<numFactors2;++i2)
                myFree(factors2[i2]);
        }
    }
    
}/* polyFactorOverpPenUlt */

void polyFactorOverIntegers(poly*& factors, int& numFactors, int*& power, const poly& f)
{
    int     i;
    double  p;
    mb      temp;
    
    if(f.array[f.deg]!=1)
        return;
    
    if(f.deg<2)
    {
        factors = (poly*)malloc(sizeof(poly));
        init(factors[0], f.deg);
        for(i=0;i<=f.deg;++i)
            factors[0].array[i] = f.array[i];
        power = (int*)malloc(sizeof(int));
        power[0] = 1;
    }
    
    // find largest in magnitude coeff.
    temp = 1;
    for(i=0;i<f.deg;++i)
        if(abs(f.array[i].num)>temp)
            temp = abs(f.array[i].num);
    temp = 2.*temp + 1.;
    while(!isItPrime(temp, 35))
        temp+=2;
    equate(p, temp);
    
    polyFactorOverpPenUlt(factors, numFactors, power, f, p);
    
}/* polyFactorOverIntegers */


//
// a(x) should be monic and belong to GF(q) where q is prime
// Q is a square matrix of dimension equal to the degree+1 of the polynomial a(x)
void polyCalcQmatrix(matq& Q, const poly& a, double q)
{
	INT32		n, i, j, m, qq;
	poly		c, ct;
	
    if(!a.deg)
        return;
    
    qq = q;
    
    n = a.deg;
    
    init(Q, n, n); // n x n matrix
    
    init(c, n-1); // c a vector with n elements, c.deg = n-1
    c.array[0] = 1;
    for(i=1;i<n;++i)
        c.array[i] = 0;
    
    j = 0;
    for(i=0;i<n;++i)
        Q.array[i][j] = c.array[i];
    
    init(ct , n-1);
    
    for(m=1;m<=qq*(n-1);++m)
    {
        ct.array[0] = -c.array[n-1] * a.array[0];
        for(i=1;i<n;++i)
            ct.array[i] = c.array[i-1] - c.array[n-1] * a.array[i];
        for(i=0;i<n;++i)
        {
            c.array[i] = ct.array[i];
            c.array[i] = c.array[i].num % q;
        }
        
        // if qq | m , j = m/qq and store c in column j of Q
        j = m / qq;
        if(m==(qq*j))
            for(i=0;i<n;++i)
                Q.array[i][j] = c.array[i];
    }
	
}/* polyCalcQmatrix */


// puts polys in rows of matrix where numRows=numPolys and numCols=maxPolyDeg+1; for smaller deg fill row end with zeroes for smaller degrees
void polySquareFree2(matq& z, const poly& x)
{
	poly		*z1;
	int		i, j, numPolys;
	
	polySquareFree(z1, numPolys, x);
	
	z.nr = numPolys;
	
	z.nc = 0;
	for(i=0;i<numPolys;i++)
	{
		if(z1[i].deg>z.nc)
			z.nc = z1[i].deg;
	}
	z.nc++;
    
    init(z, z.nr, z.nc);
	
	for(i=0;i<numPolys;i++)
	{
		for(j=0;j<z.nc;j++)
		{
			if(j<=z1[i].deg)
				z.array[i][j] = z1[i].array[j];
			else
				z.array[i][j] = 0;
		}
	}
	
}/* polySquareFree2 */


void polySFfindFactor(poly& z, const poly& a, const matq& nullVectors, double q)
{
	poly		v;
	INT32		i, k;
	bf			s;
	
	v.deg = nullVectors.nr - 1;
    init(v, v.deg);
    
	// first try 2nd column null vector; worry about high order zeroes
	//k = 1;
	for(k=1;k<nullVectors.nc;k++)
	{
		for(i=0;i<=v.deg;i++)
			v.array[i] = nullVectors.array[i][k];
		
		v = polyNormalize(v);
        makeMonic(v, v, q);
		
		for(i=0;i<q;i++)
		{
			s = i;
			z = gcd(a, v-s, q);
			if(z.deg>0)
				break;
		}
		
		if(z.deg>0)
		{
			s = z.array[z.deg];
			if(!(s.num==1))
			{
				invMod(s.num, s.num, q);
				z = s * z;
			}
			z = z % q;
			return;
		}
	}
		
}/* polySFfindFactor */


void polySFfindAllFactors(INT32& numFactors, poly*& factor, const poly& a, const matq& nullVectors, double q)
{
	INT32		i;
	poly		at, zt;
	
	at = a;
	numFactors = nullVectors.nc;
	
	factor = (poly*)malloc(numFactors*sizeof(poly));
	
	if(numFactors==1)
	{
		factor[0] = a;
		return;
	}
	
	for(i=0;i<numFactors;i++)
	{
		polySFfindFactor(zt, at, nullVectors, q);
		factor[i] = zt;
		at = at / zt;
	}

}/* polySFfindAllFactors */

// poly x should have integer coefficients and must be square free
// also the highest degree coefficient should be 1
void polySFfindTrueFactors(INT32& numFactors, poly*& factor, const poly& x)
{
	matq		Q, nullVectors, reduced;
	double		q;
	mb			qmb;
	mb			maxCoeff;
	INT32		i, numNV, swapParity;
	bool		isGood;
	
	// find max coeff of x in abs value
	maxCoeff = 0;
	for(i=0;i<=x.deg;i++)
	{
		if(abs(x.array[i].num)>maxCoeff)
			maxCoeff = abs(x.array[i].num);
	}
	
	qmb = 2.*maxCoeff + 1.;
	while(!isItPrime(qmb, 35))
		qmb+=2;
	
	equate(q, qmb);
	
	isGood = false;
	while(!isGood)
	{
		polyCalcQmatrix(Q, x, q);
		Q = Q - matqUnit(Q.nr);
		matqGaussElimModq(Q, swapParity, Q, q);
		matqNullSpace(numNV, nullVectors, Q);
		polySFfindAllFactors(numFactors, factor, x, nullVectors, q);
		isGood = true;
	}
	
	
}/* polySFfindTrueFactors */


// puts polys in rows of matrix where numRows=numPolys and numCols=maxPolyDeg+1; for smaller deg fill row end with zeroes for smaller degrees
// poly x should have integer coefficients and must be square free
// also the highest degree coefficient should be 1
void polySFfindTrueFactors2(matq& factorMatrix, const poly& x)
{
	poly		*factor;
	INT32		i, j, numFactors;
	
	polySFfindTrueFactors(numFactors, factor, x);
	factorMatrix.nr = numFactors;
	
	factorMatrix.nc = 0;
	for(i=0;i<numFactors;i++)
	{
		if(factor[i].deg>factorMatrix.nc)
			factorMatrix.nc = factor[i].deg;
	}
	factorMatrix.nc++;
	
    init(factorMatrix, factorMatrix.nr, factorMatrix.nc);
    
	for(i=0;i<numFactors;i++)
	{
		for(j=0;j<factorMatrix.nc;j++)
		{
			if(j<=factor[i].deg)
				factorMatrix.array[i][j] = factor[i].array[j];
			else
				factorMatrix.array[i][j] = 0;
		}
	}
	
}/* polySFfindTrueFactors2 */



void polyFromMatrixRow(poly& z, int theRow, const matq& x)
{
	int		j;
	
	if(theRow<0 || theRow>=x.nr)
		return;
	
	z.deg = x.nc - 1;
    init(z, z.deg);
	for(j=0;j<x.nc;j++)
		z.array[j] = x.array[theRow][j];
		
	z = polyNormalize(z);
		
}/* polyFromMatrixRow */


void findPowersOfSquareFreeFactors(matq& z1, int*& theExp, int& numPolys, const poly& x)
{
	poly		*z;
	poly		zt, r, q, temp;
	matq		z1t;
	int         i, j, maxDeg;
	bool		isGood;
	
	polySquareFree(z, numPolys, x);
	
	if(numPolys<=1)
	{
		z1.nr = 1;
		z1.nc = x.deg + 1;
        init(z1, z1.nr, z1.nc);
		
		for(i=0;i<z1.nr;i++)
			for(j=0;j<z1.nc;j++)
				z1.array[i][j] = z[i].array[j];
		
		theExp = (int*)malloc(sizeof(int));
		theExp[0] = 1;
		return;
	}
	
	theExp = (int*)malloc((numPolys-1)*sizeof(int));
	for(i=0;i<(numPolys-1);i++)
		theExp[i] = 1;
	
	for(i=1;i<numPolys;i++)
	{
		zt = z[0]; //zt contains powers of some or all of z[1], z[2], ...
		isGood = true;
		while(isGood && z[i].deg)
		{
			divEuclid(q, r, zt, z[i]);
			if(!r.deg && !r.array[0])
			{
				theExp[i-1]++;
				zt = q;
			}
			else
			{
				isGood = false;
			}
		}
	}
	
	// find max deg except for 0th poly
	maxDeg = 0;
	for(i=1;i<numPolys;i++)
	{
		if(z[i].deg>maxDeg)
			maxDeg = z[i].deg;
	}
	
	numPolys--;
	z1t.nr = numPolys;
	z1t.nc = maxDeg + 1;
    init(z1t, z1t.nr, z1t.nc);
		
	for(i=0;i<z1t.nr;i++)
	{
		for(j=0;j<z1t.nc;j++)
		{
			if(j<=z[i+1].deg)
				z1t.array[i][j] = z[i+1].array[j];
			else {
				z1t.array[i][j] = 0;
			}
		}
	}
	
	z1 = z1t;
    
    myFree(z, x.deg);
    
}/* findPowersOfSquareFreeFactors */


void findPowersOfSquareFreeFactors(matq& z, matq& theExpColM, int& numPolys, const poly& x)
{
	int		*theExp, i;
	
	findPowersOfSquareFreeFactors(z, theExp, numPolys, x);
	
    init(theExpColM, z.nr, 1);
	
	for(i=0;i<theExpColM.nr;i++)
		theExpColM.array[i][0] = theExp[i];
	
}/* findPowersOfSquareFreeFactors */

// does gcd of a(x) and modified v(x), a null poly vector
// it returns true and z, a factor of a(x), or false and z = 1
// p must be prime
bool polyFindOneFactor(poly& z, const poly& a, const poly& v, double p)
{
    int     i;
    bf      s;
    poly    zt, q, r;
    bool    isFactor;
    
    //cout << "a.deg = " << a.deg << endl;  // v.derg = 13 wrong
    //if(v.array[0]==0)
        //polyReduce(v);
    isFactor = false;
    for(i=0;i<p;++i)
    {
        s = i;
        gcd(zt, a, v-s, p);
        //cout << "zt = " << zt << endl;
        if(zt.deg>0)
        {
            divEuclid(q, r, a, zt, p);
            //cout << "r = " << r << endl;
            //r = polyNormalize(r);
            if(r.deg==0 && r.array[0]==0)
            {
                isFactor = true;
                break;
            }
        }
    }
    
    if(isFactor)
        z = zt;
    
    return isFactor;
    
}/* polyFindOneFactor */

bool polyFindAllFactors(poly*& factors, const poly& a, poly* polyNV, int& numFactors, int numNV, double q)
{
    poly    aFactor, at, v;
    int     i;
    bool    noneFound;
    
    at = a;
    factors = (poly*)malloc(numNV*sizeof(poly));
    for(i=0;i<numNV;++i)
        init(factors[i]);
    
    numFactors = 0;
    
    while(at.deg>0)
    {
        noneFound = true;
        for(i=1;i<numNV;++i)  // skip 0th vector
        {
            v = polyNV[i];
            makeMonic(v, v, q);
            if(polyFindOneFactor(aFactor, at, v, q))
            {
                factors[numFactors++] = aFactor;
                div(at, at, aFactor, q);
                noneFound = false;
                break;
            }
        }
        if(noneFound)
            return false;
    }
    
    return true;

}/* polyFindAllFactors */

// for fpPlugin use matq to hold polys and their powers by rows
// each row from left to right holds power, deg, coeffs.
// so nr = numPolys, nc = degMax + 3
// polyFactorOverp(poly*& factors, int& numFactors, int*& power, const poly& f, double p)
void polyFactorOverpForPlugin(matq& z, const poly& f, double p)
{
    poly    *factors;
    int     i, j, numFactors, *power, degMax;
    
    if(f.array[f.deg]!=1)
        return;
    
    if(f.deg<2)
    {
        numFactors = 1;
        factors = (poly*)malloc(sizeof(poly));
        init(factors[0], f.deg);
        for(i=0;i<=f.deg;++i)
            factors[0].array[i] = f.array[i];
        power = (int*)malloc(sizeof(int));
        power[0] = 1;
    }
    
    if(f.deg>1)
        polyFactorOverpPenUlt(factors, numFactors, power, f, p);
    
    // find degMax
    degMax = 1;
    for(i=0;i<numFactors;++i)
        if(factors[i].deg>degMax)
            degMax = factors[i].deg;
    
    init(z, numFactors, degMax+3);
    for(i=0;i<z.nr;++i)
    {
        z.array[i][0] = power[i];
        z.array[i][1] = factors[i].deg;
        for(j=2;j<=factors[i].deg+2;++j)
            z.array[i][j] = factors[i].array[j-2];
    }
    
}/* polyFactorOverpForPlugin */

// for fpPlugin, gets poly and power at row
void polyFromMatq(poly& z, int& power, const matq& x, int row)
{
    int     i, deg;
    equate(power, x.array[row][0].num);
    equate(deg, x.array[row][1].num);
    
    init(z, deg);
    for(i=0;i<=+deg;++i)
        z.array[i] = x.array[row][i+2];
    
}/* polyFromMatq */


// each row of z has power, deg, poly from low to high
void polyArrayToMatq(matq& z, const poly*& x, const int* power, int numPolys)
{
    int         i, j, degMax;
    
    // find degMax
    degMax = 0;
    for(i=0;i<numPolys;++i)
        if(x[i].deg>degMax)
            degMax = x[i].deg;
    
    init(z, numPolys, degMax+3);
    
    for(i=0;i<z.nr;++i)
    {
        z.array[i][0] = power[i];
        z.array[i][1] = x[i].deg;
        
        for(j=2;j<=x[i].deg+2;++j)
            z.array[i][j] = x[i].array[j-2];
    }
    
}/* polyArrayToMatq */

// converts poly array as x format into zString
void polyArrayToString(char*& zString, const poly* x, const int* power, int numPolys)
{
    int     i, length;
    char    *xString, numString[10];
    poly    temp;
    
    length = 0;
    for(i=0;i<numPolys;++i)
    {
        temp = x[i];
        //cout << "x[i] = " << x[i] << endl;
        polyConvToxString(xString, temp);
        //cout << xString << endl;
        length+=strlen(xString);
        free(xString);
    }
    length+=(14*numPolys);  // for extras
    
    zString = (char*)malloc(length*sizeof(char));
    zString[0] = 0;
    
    for(i=0;i<numPolys;++i)
    {
        temp = x[i];
        polyConvToxString(xString, temp);
        if(power[i]>1)
        {
            strcat(zString, "(");
            strcat(zString, xString);
            strcat(zString, ")^");
            sprintf(numString,"%d", power[i]);
            strcat(zString, numString);
        }
        else
            strcat(zString, xString);
        strcat(zString, "\n");
    }
    
}/* polyArrayToString */

// Over GF(p^m) there are exactly φ(p^m − 1)/m primitive polynomials of degree m, where φ is Euler's totient function.
void numPrimPolys(mb& z, const mb& p, const mb& m)
{
    mb      q, zt;
    
    q = power(p, m);
    zt = phi(q-1) / m;
    z = zt;
    
}/* numPrimPolys */

mb numPrimPolys(const mb& p, const mb& m)
{
    mb  z;
    
    numPrimPolys(z, p, m);
    
    return z;
    
}/* numPrimPolys */

void polyFactorOverp(poly*& factors, int& numFactors, int*& power, const poly& f, double p)
{
    int     i, jIR, jR;
    poly    *factorsIR, *factorsR, temp;
    int     *powerIR, *powerR, numFactorsIR, numFactorsR;
    int     j;
    
    if(f.array[f.deg]!=1)
    {
        numFactors = 0;
        return;
    }
    
    if(f.deg<2)
    {
        numFactors = 1;
        factors = (poly*)malloc(sizeof(poly));
        init(factors[0], f.deg);
        for(i=0;i<=f.deg;++i)
            factors[0].array[i] = f.array[i];
        power = (int*)malloc(sizeof(int));
        power[0] = 1;
        return;
    }
    
    polyFactorOverpPenUlt(factors, numFactors, power, f, p);
    /*
    for(j=0;j<numFactors;++j)
        cout << factors[j] << " power = " << power[j] << endl;
    cout << endl;
    */
    
    if(numFactors<2)
        return;
    
    factorsIR = (poly*)malloc(f.deg*sizeof(poly));
    factorsR = (poly*)malloc(f.deg*sizeof(poly));
    powerIR = (int*)malloc(f.deg*sizeof(int));
    powerR = (int*)malloc(f.deg*sizeof(int));
    numFactorsIR = numFactorsR = 0;
    
    for(i=0;i<f.deg;++i)
    {
        init(factorsIR[i]);
        init(factorsR[i]);
    }
    
    jIR = jR = 0;
    // do initial distribution of red. & irrerd. polys from factors and power
    for(i=0;i<numFactors;++i)
    {
        temp = factors[i];
        if(temp.deg<2)
        {
            if(temp.deg<1)
                continue;
            else
            {
                // temp.deg=1 so irred.
                factorsIR[jIR] = temp;
                powerIR[jIR++] = power[i];
                numFactorsIR++;
            }
        }
        else
        {
            // temp.deg>1 so could be reducible
            factorsR[jR] = temp;
            powerR[jR++] = power[i];
            numFactorsR++;
        }
        myFree(temp);
    }
    
    myFree(factors, numFactors);
    free(power);
    
    if(!numFactorsR)
    {
        // we're done
        // sort from low degree to high
        numFactors = numFactorsIR;
        
        int     deg, degMax;
        degMax = 0;
        for(i=0;i<numFactors;++i)
            if(factorsIR[i].deg>degMax)
                degMax = factorsIR[i].deg;
        
        factors = (poly*)malloc(numFactors*sizeof(poly));
        power = (int*)malloc(numFactors*sizeof(int));
        
        j = 0;
        for(deg=1;deg<=degMax;++deg)
        {
            for(i=0;i<numFactors;++i)
            {
                if(factorsIR[i].deg==deg)
                {
                    init(factors[j]);
                    factors[j] = factorsIR[i];
                    power[j++] = powerIR[i];
                }
            }
        }
        myFree(factorsIR, f.deg);
        myFree(factorsR, f.deg);
        free(powerIR);
        free(powerR);
        return;
    }
    
    // now factorsR is not empty
    while(numFactorsR>0)
    {
        temp = factorsR[0];  // always investigate the top poly in factorsR
        polyFactorOverpPenUlt(factors, numFactors, power, temp, p);
        if(numFactors==1)
        {
            // temp is irreducible
            factorsIR[jIR] = temp;
            powerIR[jIR++] = powerR[0];
            numFactorsIR++;
            //cout << "temp = " << temp << endl;
            myFree(temp);
            // since top poly was irreducible, we need to move polys up one in factorsR and move up in powerR
            for(i=1;i<numFactorsR;++i)
            {
                myFree(factorsR[i-1]);
                factorsR[i-1] = factorsR[i];
            }
            numFactorsR--;
            jR--;
            if(!numFactorsR)
            {
                // we are done
                 // sort from low degree to high
                myFree(factors, numFactors);
                free(power);
                numFactors = numFactorsIR;
                factors = (poly*)malloc(numFactors*sizeof(poly));
                power = (int*)malloc(numFactors*sizeof(int));
                
                int     deg, degMax;
                degMax = 0;
                for(i=0;i<numFactors;++i)
                    if(factorsIR[i].deg>degMax)
                        degMax = factorsIR[i].deg;
                
                j = 0;
                for(deg=1;deg<=degMax;++deg)
                {
                    for(i=0;i<numFactors;++i)
                    {
                        if(factorsIR[i].deg==deg)
                        {
                            init(factors[j]);
                            factors[j] = factorsIR[i];
                            power[j++] = powerIR[i];
                        }
                    }
                }
                myFree(factorsIR, f.deg);
                myFree(factorsR, f.deg);
                free(powerIR);
                free(powerR);
                return;
            } // if
        }
        else
        {
            // numFactors>1, so replace top poly in factorsR with factors[0] and put rest of factors at bottom of factorsR
            //numFactorsR++;
            myFree(temp);
            myFree(factorsR[0]);
            factorsR[0] = factors[0];
            // powerR[0] is already correct
            for(i=1;i<numFactors;++i)
            {
                myFree(factorsR[jR]);
                factorsR[jR] = factors[i];
                powerR[jR++] = powerR[0];
                numFactorsR++;
            }
        } // else
        myFree(factors, numFactors);
        free(power);
    } // while
    
}/* polyFactorOverp */

void polyFactorOverp(char*& outString, const poly& f, double p)
{
    poly    *factors, pCheck, zCheck;
    int     i, j, *power, numPolys, length;
    char    *stringp, *stringz, *zString, numString[10];
    poly    temp;
    
    polyFactorOverp(factors, numPolys, power, f, p);
    polyArrayToString(zString, factors, power, numPolys);
    length = strlen(zString) + 25;
    pCheck = 1;
    zCheck = 1;
    
    for(i=0;i<numPolys;++i)
    {
        for(j=0;j<power[i];++j)
        {
            temp = factors[i] % p;
            pCheck = (pCheck * temp) % p;
            zCheck = zCheck * factors[i];
        }
    }
    
    polyConvToxString(stringp, pCheck);
    polyConvToxString(stringz, zCheck);
    length = length + strlen(stringp) + strlen(stringz) + 300;
    
    outString = (char*)malloc(length*sizeof(char));
    strcpy(outString, "Irreducible Factors = ");
    sprintf(numString, "%d", numPolys);
    strcat(outString, numString);
    strcat(outString, "\n");
    strcat(outString, zString);
    strcat(outString, "\n");
    strcat(outString, "product over GF(p):\n");
    strcat(outString, stringp);
    strcat(outString, "\n\n");
    strcat(outString, "product over Z:\n");
    strcat(outString, stringz);
    strcat(outString, "\n");
    
    myFree(factors, numPolys);
    free(power);
    free(stringp);
    free(stringz);
    free(zString);
    
}/* polyFactorOverp */

// Over GF(p^m) there are exactly φ(p^m − 1)/m primitive polynomials of degree m, where φ is Euler's totient function.
// A primitive poly of degree m is a factor of x^(p^m - 1) and not a factor of x^n - 1 n = p^m - 2, p^m - 3, ..., m
bool findPrimitivePolys(poly*& z, int& numPolys, int m, double p)
{
    int     i, j, numPrim, numDegM, numFactors, *power, q;
    poly    f, *factors, *factorsCand;
    bf      one=1;
    mb      mbM, mbP,  mbNumPrim;
    poly    temp, qt, rt;
    char    *xString;
    int     n, N;
    
    if(m<=0)
    {
        numPolys = 0;
        return false;
    }
    
    if(m==1)
    {
        // x + 1 is primitive for all primes p
        numPolys = 1;
        z = (poly*)malloc(sizeof(poly));
        init(z[0], 1);
        z[0].array[0] = 1;
        z[0].array[1] = 1;
        return true;
    }
    
    //mb numPrimPolys(const mb& p, const mb& m)
    mbP = p;
    mbM = m;
    mbM = numPrimPolys(mbP, mbM);
    equate(numPrim, mbM);
    
    q = pow(p, m);
    N = q - 1;
    f = xTon(N) - one;
    polyFactorOverp(factors, numFactors, power, f, p);
    
    // How many polys in factors have degree m?
    numDegM = 0;
    for(i=0;i<numFactors;++i)
        if(factors[i].deg==m)
            numDegM++;
    
    if(numDegM<numPrim)
        return false;
    
    // copy polys of degree m into factorsCand
    factorsCand = (poly*)malloc(numDegM*sizeof(poly));
    for(i=0;i<numDegM;++i)
        init(factorsCand[i]);
    j = 0;
    for(i=0;i<numFactors;++i)
        if(factors[i].deg==m)
            factorsCand[j++] = factors[i];
    
    z = (poly*)malloc(numDegM*sizeof(poly));
    for(i=0;i<numDegM;++i)
        init(z[i]);
    
    bool    isPrim;
    // we now check whether each poly in factorsCand divides x^n - 1, n = p^m - 2, p^m - 3, ..., m
    j = 0;
    numPolys = 0;
    for(i=0;i<numDegM;++i)
    {
        temp = factorsCand[i];
        isPrim = true;
        for(n=q-2;n>1;n--)
        {
            if(n*(N/n)==N)
            {
                f = xTon(N/n) - one;
                divEuclid(qt, rt, f, temp, p);
                if(rt==0)
                {
                    isPrim = false;
                    break;
                }
                myFree(qt);
                myFree(rt);
            }
        }
        if(isPrim)
        {
           // cout << "i = " << i << endl;
            polyConvToxString(xString, temp);
            //cout << xString << endl << endl;
            free(xString);
            z[j++] = temp;
            numPolys++;
        }
    }
    if(numPolys==numPrim)
        return true;
    return false;
    
}/* findPrimitivePolys */

bool findPrimitivePolys(char*& outString, int m, double p)
{
    poly        *z;
    int         i, numPolys, *power;
    
    if(!findPrimitivePolys(z, numPolys, m, p))
        return false;
    power = (int*)malloc(numPolys*sizeof(int));
    for(i=0;i<numPolys;++i)
        power[i] = 1;
    
    polyArrayToString(outString, z, power, numPolys);
    
    return true;
    
}/* findPrimitivePolys */

// like polys are bunched together
static void polyConvertToPower(poly*& z, int*& power, int& numFactors, const poly* zt, const int numFactors2)
{
    int     i, j, jj, n;
    poly    temp, temp1;
    
    n = numFactors2;
    z = (poly*)malloc(n*sizeof(poly));
    power = (int*)malloc(n*sizeof(int));
    for(i=0;i<n;++i)
    {
        init(z[i]);
        power[i] = 0;
    }
    
    
    i = 0; // i points to poly in zt
    j = 0; // j is z index
    while(true)
    {
        temp = zt[i++];
        jj = j;
        z[j++] = temp;
        power[jj]+=1;
        if(i==numFactors2) // don't have a following polynomial to one just stored in z
        {
            numFactors = j;
            return;
        }
        temp1 = zt[i];  // there is a following polynomial, but what if it's last?
        // it can be same or different
        while(temp1==temp)
        {
            power[jj]+=1;
            if(i<numFactors2-1)
                temp1 = zt[++i];
            else
            {
                // we are done
                numFactors = j;
                return;
            }
        }
    }
}/* polyConvertToPower */

// x must be monic integer poly
// if isGood=false last poly in z may not be irreducible
void polyFactorOverIntegers(poly*& z, int*& power, int& numFactors, bool& isGood, const poly& x)
{
    poly        *zt, *z1, xt, q, r, t;
    int         i, j, k, m, numFactors1, numFactors2, *power1;
    mb          temp;
    double      p;
    int         count, countMax=10;
    
    if(!isPolyIntMonic(x))
        return;
    m = x.deg;
    // x.deg is the max number of factors
    z1 = (poly*)malloc(m*sizeof(poly));
    for(i=0;i<m;++i)
        init(z1[i]);
    
    // calculate p as greater than twice largest abs value of coeffs.
    temp = 0;
    for(i=0;i<=x.deg;++i)
        if(abs(x.array[i].num)>temp)
            temp = abs(x.array[i].num);
            
            temp = 2.*temp + 1;
    while(!isItPrime(temp, 35))
        temp+=2;
    equate(p, temp);
    
    xt = x;
    k = 0; // indexes polys stored in z poly array
    for(count=0;count<countMax;++count)
    {
        polyFactorOverp(zt, numFactors2, power1, xt, p);
        for(i=0;i<numFactors2;++i)
        {
            t = zt[i];
            for(j=0;j<power1[i];++j)
            {
                divEuclid(q, r, xt, t);
                if(r==0)
                {
                    z1[k++] = t;
                    xt = q;
                    if(xt==1)
                    {
                        numFactors1 = k;
                        polyConvertToPower(z, power, numFactors, z1, numFactors1);
                        for(i=0;i<numFactors1;++i)
                            myFree(z1[i]);
                        free(z1);
                        for(i=0;i<numFactors2;++i)
                            myFree(zt[i]);
                        free(zt);
                        isGood = true;
                        return;
                    }
                }
            }
        }
        for(i=0;i<numFactors2;++i)
            myFree(zt[i]);
        free(zt);
        p+=2;
        while(!isItPrime(p, 35))
            p+=2;
    }
    //reached count end
    z1[k++] = xt;
    numFactors1 = k;
    polyConvertToPower(z, power, numFactors, z1, numFactors1);
    for(i=0;i<numFactors1;++i)
        myFree(z1[i]);
    free(z1);
    isGood = false;
    
}/* polyFactorOverIntegers */


void polyFactorOverIntegers(char*& outString, const poly& x)
{
    poly        *z;
    int         i, j, *power, numFactors, length;
    bool        isGood;
    char        *zString, numString[10];
    
    polyFactorOverIntegers(z, power, numFactors, isGood, x);
    polyArrayToString(zString, z, power, numFactors);
    length = strlen(zString) + 300;
    outString = (char*)malloc(length*sizeof(char));
    outString[0] = 0;
    
    if(isGood)
    {
        strcpy(outString, "Irreducible Factors = ");
        sprintf(numString, "%d", numFactors);
        strcat(outString, numString);
        strcat(outString, "\n");
        strcat(outString, zString);
        strcat(outString, "\n");
        free(zString);
        return;
    }
    
    strcpy(outString, "Irreducible Factors = ");
    sprintf(numString, "%d", numFactors-1);
    strcat(outString, numString);
    strcat(outString, " or ");
    sprintf(numString, "%d", numFactors);
    strcat(outString, numString);
    strcat(outString, "\n");
    strcat(outString, zString);
    strcat(outString, "\n");
    strcat(outString, "The last polynomial may not be irreducible.\n");
    free(zString);

}/* polyFactorOverIntegers */

