/*
 *  mbMod.cpp
 *  fp
 *
 *  Created by Robert Delaney on 4/24/09.
 *  Copyright 2009 __Bob Delaney's Science Software__. All rights reserved.
 *
 */

#include "fp.h"
#include "bf.h"

using namespace std;

// returns theBit (0 or 1) at bitPosition in blockNum of x
static INT32 findBit(const mb& x, INT32 bitPosition, INT32 blockNum)
{
	UINT32				mask;
	
	// now construct mask for getting theBit, we want to AND with something like 00010000
	mask = 1;
	mask = (mask<<bitPosition);
	return (mask & x.b[blockNum]);
	
}/* findBit */


// Finds z = b^(m*2^k)(mod n) where k>=0
// Uses p(mod n)*q(mod n) = p*q(mod n)
void powerMod(mb& z, const mb& b, const mb& m, INT32 k, const mb& n)
{
	static mb		one;
	mb				bmod, zt, zTemp, mt;
	static bool		initGood=false;
	INT32			i, blockNum, bitPosition;
	
	if(!initGood)
	{
		init(one, 1);
		initGood = true;
	}
	
	if(m.n<0)
		return;
	
	if(n.n<=0)
	{
		z.n = 0;
		return;
	}
	
	if(!b.n	&& !m.n)
	{
		z.n = 0;
		return;
	}
	
	if(!compare(n, one))
	{
		z.n = 0;
		return;
	}
	
	if(!m.n || !compare(b, one))
	{
		equate(z, one);
		return;
	}
	   
	if(!b.n)
	{
		z.n = 0;
		return;
	}
	
	equate(mt, m);
	
	divRem(zTemp, bmod, b, n); // bmod = b(mod n)
	
	if(k<0)
	{
		mbShiftRight(mt, mt, -k); // x 2^(-|k|)
		if(!mt.n)
		{
			equate(z, one);
			return;
		}
	}
	else
	{
		// we build up to b^(2^k)(mod n)
		for(i=0;i<k;i++)
		{
			mul(bmod, bmod, bmod); // accumulates b^(2^i)(mod n)
			divRem(zTemp, bmod, bmod, n); // bmod = bmod(mod n)
		}
	}
	
	// now bmod = b^(2^k)(mod n), so need to build to b^(m*2^k)(mod n)
	// using mb form of m in binary
	equate(zt, 1.);  // value if m=0
	blockNum = 0;
	while(blockNum<m.n)
	{
		bitPosition = 0;
		while(bitPosition<32)
		{
			if(findBit(m, bitPosition, blockNum))
			{
				// bit = 1
				mul(zt, zt, bmod);
				divRem(zTemp, zt, zt, n); // reduce zt
			}
			mul(bmod, bmod, bmod);
			divRem(zTemp, bmod, bmod, n); // update and reduce bmod so k->k+1 etc.
			bitPosition++;
		}
		blockNum++;
	}
	
	equate(z, zt);

}/* powerMod */


mb& powerMod(const mb& b, const mb& m, INT32 k, const mb& n)
{
	static mb		z;
	
	powerMod(z, b, m, k, n);
	return z;
	
}/* powerMod */


// if returns true then n is composite; if returns false it's inconclusive
bool Miller_Rabin(const mb& n)
{
	static mb		one, minusone;
	mb				m, b, nn, zMod;
	static bool		initGood=false;
	INT32			i, q;
	
	if(!initGood)
	{
		init(one, 1);
        init(minusone, -1);
		initGood = true;
	}
	
	sub(nn, n, 1.); // nn = n-1 (mod value equivalent to -1)
	
	// get random 1 <= b <= n-1
	mbRandom(b, nn); // 0 <= b <= n-2
	add(b, b, 1.); // 1 <= b <= n-1
	
	// find q and m so n-1 = m*2^q, m an odd integer
	equate(m, nn);
	q = 0;
	while(isItEven(m))
	{
		mbShiftRight(m, m, 1); // m=m/2
		q++;
	}
	
	// case a, if b^m = 1 (mod n) then return false
	powerMod(zMod, b, m, 0, n);
	if(!compare(zMod, one))
		return false;
	
	// case b, if there is an integer i in [0, q-1] such that b^(m*2^i) = nn (mod n) then return false
	for(i=0;i<q;i++)
	{
		//powerMod(zMod, b, m, i, n);
		if(!compare(zMod, nn) || !compare(zMod, minusone))
			return false;
		myModulus(zMod, zMod*zMod, n);
	}
	
	return true;
	
}/* Miller_Rabin */


bool isItPrime(const mb& x, INT32 numTrials)
{
	INT32		i;
	
	if(x==1)
		return false;
	
	if(x<=0.)
		return false;
	
	for(i=0;i<numTrials;i++)  // For 30 trials that probability is less than about 8.7e-19 ((1/4)^30).
	{
		if(Miller_Rabin(x))
			return false;
	}
	return true;

}/* isItPrime */

bool isItPrime(int x, INT32 numTrials)
{
    mb      xmb;
    
    xmb = x;
    
    return isItPrime(xmb, numTrials);
    
}/* isItPrime */

/*
	z = x % n ; x can be negative, but let's always put out positive mods or zero
	If x>=0 just use divRem.
	if x<0 then for n>0 rem<0, so rem = rem + n > 0
*/
void myModulus(mb& z, const mb& x, const mb& n)
{
	static mb		temp, nn;
	static bool		initGood=false;
		
	if(!initGood)
	{
		init(temp);
		initGood = true;
	}
	
	if(!n.n)
		return;
	
	if(!x.n)
	{
		z.n = 0;
		return;
	}
	
	nn = abs(n);
	
	divRem(temp, z, x, nn);
	
	if(x.n<0 && nn.n>0)
		add(z, z, nn);
	
	if(z>nn/2)
		z = z - nn;
	 
	/*
	if(z==nn)
		z = 0;
     */
		
}/* myModulus */


/*
 z = x % n ; x can be negative, but let's always put out positive mods
 If x>=0 just use divRem.
 if x<0 then for n>0 rem<0, so rem = rem + n > 0
 */
void myModulus(mb& z, const mb& x, double n)
{
	static mb		temp, nt;
	static bool		initGood=false;
	
	if(!initGood)
	{
		init(temp);
		init(nt);
		initGood = true;
	}
	
	if(n<=0)
	{
		z = 0;
		return;
	}
	
	if(!x.n)
	{
		z = 0;
		return;
	}
	
	equate(nt, n);
	
	divRem(temp, z, x, nt);
	
	if(x.n<0 && nt.n>0)
		add(z, z, nt);
	
	if(z>nt/2)
		z = z - nt;
	 
	/*
	if(z==nt)
		z = 0;
    */
	
}/* myModulus */


// http://www.mathreference.com/num-mod,minv.html
// p should be prime
// s = yt, t = xt
// returns false if x has no inverse mod, else z = x^-1 mod p
bool invMod(mb& z, const mb& x, const mb& p)
{
	mb		s, t, q, a, b, m, r, temp, one=1; // xt = u  yt = v
    mb      xt;
    
	if(x==1)
	{
		z = one;
		return true;
	}
    
    xt = x;
    if(xt.n<0)
        while(xt.n<0)
            xt = xt + p;
    
    if(!xt || p<=1.)
    {
        z = 0;
        return true;
    }
    
    s = p;
    t = xt;
    m = p;
    a = 0;
    b = 1.;
	
	while(t!=0)
	{
		divRem(q, r, s, t); // r = s % t
		s = t;
		t = r;
		temp = (a-b*q) % m;
		a = b;
		b = temp;
	}
	
	if(compare(s, one))
		return false;
	
	z = a;
	return true;
	
}/* invMod */

bool invMod(mb& z, const mb& x, double p)
{
    mb      pmb;
    
    equate(pmb, p);
    return invMod(z, x, pmb);
    
}/* invMod */

mb invMod(const mb& x, double p)
{
    mb  z;
    invMod(z, x, p);
    return z;
    
}/* invMod */

// https://en.wikipedia.org/wiki/Modular_arithmetic
// If c = d mod phi(n), where phi is Euler’s totient function, then a*c = a^d mod n provided gcd(a, n) = 1.
/*void phiInt(int& z, double n)
{
    bf     phi,temp;
    mb     p;
    int    nn;
    
    nn = n;
    
    if(isItPrime(nn, 35))
    {
        phi = nn - 1;
        equate(z, phi.num);
        return;
    }
    
    if(!nn)
    {
        z = 0;
        return;
    }
    
    if(nn==1 ||nn==2)
    {
        z = 1;
        return;
    }
    // nn >= 4 and not prime now
    if(2*(nn/2)==nn)
    {
        phi.num = 1;
        phi.den = 2;
    }
    else
        phi = 1;
    
    for(p=3;p<=nn/2+1;p+=2)
    {
        if(isItPrime(p, 35))
        {
            if(p*(n/p)==n)
            {
                temp = p;
                phi = phi * (1 - 1/temp);
            }
        }
    }
    phi = phi * n;
    equate(z, phi.num);
    return;
    
}*//* phiInt */

/*
int phiInt(double n)
{
    int  z;
    
    phiInt(z, n);
    
    return z;
    
}*//* phiInt */


// https://en.wikipedia.org/wiki/Modular_arithmetic
// If c = d mod phi(n), where phi is Euler’s totient function, then a*c = a^d mod n provided gcd(a, n) = 1.
void phi(mb& z, const mb& n)
{
    bf     phi,temp;
    mb     p;
    
    if(isItPrime(n, 35))
    {
        phi = n - 1;
        z = phi.num;
        return;
    }
    
    if(!n)
    {
        z = 0;
        return;
    }
    
    if(n==1 ||n==2)
    {
        z = 1;
        return;
    }
    // n >= 4 and not prime now
   if(2*(n/2)==n)
   {
       phi.num = 1;
       phi.den = 2;
   }
    else
        phi = 1;
  
    for(p=3;p<=n/2+1;p+=2)
    {
        if(isItPrime(p, 35))
        {
            if(p*(n/p)==n)
            {
                temp = p;
                phi = phi * (1 - 1/temp);
            }
        }
    }
    phi = phi * n;
    z = phi.num;
    return;
    
}/* phi */

mb phi(const mb& n)
{
    mb  z;
    
    phi(z, n);
    
    return z;
    
}/* phi */
