/*
 *  matqMath.cpp
 *  fp
 *
 *  Created by Robert Delaney on 5/28/13.
 *  Copyright 2013 Bob Delaney's Science Software. All rights reserved.
 *
 */


#include "fp.h"
#include "mb.h"
#include "bf.h"
#include "matq.h"
#include "matqMath.h"
#include "matqConv.h"
#include "poly.h"
#include "polyMath.h"
#include "newRandom.h"
#include "matc.hpp"
#include "matcConv.hpp"
#include "matcMath.hpp"

// swap row i with row j
static void matqSwapRows(matq& z, const matq& x, INT32 i, INT32 j)
{
    matq    xt;
	bf		temp;
	INT32	k;
    
    xt = x;
	
	for(k=0;k<x.nc;k++)
	{
		temp = xt.array[i][k];
		xt.array[i][k] = xt.array[j][k];
		xt.array[j][k] = temp;
	}
    
    z = xt;
	
}/* matqSwapRows */

static bool doGauss(matq& x, INT32& swapParity, INT32 ii, INT32 jj)
{
    bf   temp, temp1;
    INT32       i, j, k;
    bool        isZero;
    
    if(ii<0 || jj<0 || ii>=x.nr || jj>=x.nc)
        return false;
    
    k = 0;
    
    // is pivot zero?
    if(!x.array[ii][jj].num)
    {
        // must swap rows
        // find non-zero element below pivot in pivot column
        isZero = true;
        for(i=ii+1;i<x.nr;++i)
        {
            if(x.array[i][jj].num!=0)
            {
                isZero = false;
                k = i;
                break;
            }
        }
        
        if(isZero)
            return false;
        
        // have non-zero element in row k, so swap rows ii with k
        matqSwapRows(x, x, ii, k);
        swapParity = -swapParity;
    }
    
    // now we do the Gauss Elim. operation
    // pivot element xt[ii][jj] is not zero
    
    // now for every k not ii subtract row ii multiplied by xt[k][jj] from row k provided x[k][jj] != 0
    for(k=0;k<x.nr;++k)
    {
        if(k!=ii && (x.array[k][jj].num!=0))
        {
            //temp = x[k][jj] / x[ii][jj];
            //DivCC(temp, x[k][jj], x[ii][jj]);
            temp = x.array[k][jj] / x.array[ii][jj];
            for(j=0;j<x.nc;++j)
            {
                //MulCC(temp1, x[ii][j], temp);
                temp1 = x.array[ii][j] * temp;
                //SubCC(x[k][j], x[k][j], temp1);
                x.array[k][j] = x.array[k][j] - temp1;
            }
        }
    }
    return true;
    
}/* doGauss */

static bool doGaussModq(matq& z, const matq& x, INT32& swapParity, INT32 ii, INT32 jj, INT32 q)
{
    matq        xt;
    bf          temp, temp1;
    bf          tempInv;
    INT32       i, j, k;
    bool        isZero;
    
    if(ii<0 || jj<0 || ii>=x.nr || jj>=x.nc)
        return false;
    
    k = 0;
    xt = x;
    
    // is pivot zero?
    if(!xt.array[ii][jj].num)
    {
        // must swap rows
        // find non-zero element below pivot in pivot column
        isZero = true;
        for(i=ii+1;i<xt.nr;++i)
        {
            if(xt.array[i][jj].num!=0)
            {
                isZero = false;
                k = i;
                break;
            }
        }
        
        if(isZero)
            return false;
        
        // have non-zero element in row k, so swap rows ii with k
        matqSwapRows(xt, xt, ii, k);
        swapParity = -swapParity;
    }
    
    // now we do the Gauss Elim. operation
    // pivot element xt[ii][jj] is not zero
    
    // now for every k not ii subtract row ii multiplied by xt[k][jj] from row k provided x[k][jj] != 0
    for(k=0;k<x.nr;++k)
    {
        if(k!=ii && (xt.array[k][jj].num!=0))
        {
            //temp = x[k][jj] / x[ii][jj];
            //DivCC(temp, x[k][jj], x[ii][jj]);
            tempInv = xt.array[ii][jj];
            invMod(tempInv.num, tempInv.num, q);
            tempInv = tempInv.num;
            temp = xt.array[k][jj] * tempInv;
            temp = temp.num % q;
            for(j=0;j<x.nc;++j)
            {
                //MulCC(temp1, x[ii][j], temp);
                temp1 = xt.array[ii][j] * temp;
                temp1 = temp1.num % q;
                //SubCC(x[k][j], x[k][j], temp1);
                xt.array[k][j] = xt.array[k][j] - temp1;
                xt.array[k][j] = xt.array[k][j].num % q;
            }
        }
    }
    z = xt;
    
    return true;
    
}/* doGaussModq */

static bool doReduce(matq& x, INT32 ii, INT32 jj)
{
    bf              temp, temp1;
    INT32            i, j, k;
    bool            isZero;
    
    if(ii<0 || jj<0 || ii>=x.nr || jj>=x.nc)
        return false;
    
    k = 0; // to avoid incorrect warning
    // is pivot zero?
    if(x.array[ii][jj].num==0)
    {
        // must swap rows
        // find non-zero element below pivot in pivot column
        isZero = true;
        for(i=ii+1;i<x.nr;++i)
        {
            if(x.array[i][jj].num!=0)
            {
                isZero = false;
                k = i;
                break;
            }
        }
        
        if(isZero)
            return false;
        
        // have non-zero element in row k, so swap rows ii with k
        matqSwapRows(x, x, ii, k);
    }
    // now we do the reduce operation
    // pivot element xt.array[ii][jj] is not zero
    // divide each element of row ii by xt.array[ii][jj] making xt.array[ii][jj] = 1
    //temp.Re = x[ii][jj].Re;
    //temp.Im = x[ii][jj].Im;
    temp = x.array[ii][jj];
    for(j=0;j<x.nc;++j)
    {
        //DivCC(x[ii][j], x[ii][j], temp);
        x.array[ii][j] = x.array[ii][j] / temp;
    }
    
    // now for every k not ii subtract row ii multiplied by xt[k][jj] from row k provided x[k][jj] != 0
    for(k=0;k<x.nr;++k)
    {
        if(k!=ii && x.array[k][jj].num!=0)
        {
            //temp.Re = x[k][jj].Re;
            //temp.Im = x[k][jj].Im;
            temp = x.array[k][jj];
            for(j=0;j<x.nc;++j)
            {
                //MulCC(temp1, x[ii][j], temp);
                //SubCC(x[k][j], x[k][j], temp1);
                temp1 = x.array[ii][j] * temp;
                x.array[k][j] = x.array[k][j] - temp1;
            }
        }
    }
    return true;
    
}/* doReduce */

static bool doReduceModp(matq& x, INT32 ii, INT32 jj, double p)
{
    bf              temp, temp1;
    INT32            i, j, k;
    bool            isZero;
    
    if(ii<0 || jj<0 || ii>=x.nr || jj>=x.nc)
        return false;
    
    k = 0; // to avoid incorrect warning
    // is pivot zero?
    if(x.array[ii][jj].num==0)
    {
        // must swap rows
        // find non-zero element below pivot in pivot column
        isZero = true;
        for(i=ii+1;i<x.nr;++i)
        {
            if(x.array[i][jj].num!=0)
            {
                isZero = false;
                k = i;
                break;
            }
        }
        
        if(isZero)
            return false;
        
        // have non-zero element in row k, so swap rows ii with k
        matqSwapRows(x, x, ii, k);
    }
    // now we do the reduce operation
    // pivot element xt.array[ii][jj] is not zero
    // divide each element of row ii by xt.array[ii][jj] making xt.array[ii][jj] = 1
    //temp.Re = x[ii][jj].Re;
    //temp.Im = x[ii][jj].Im;
    temp = x.array[ii][jj];
    temp.num = invMod(temp.num, p);
    temp = temp.num;
    for(j=0;j<x.nc;++j)
    {
        //DivCC(x[ii][j], x[ii][j], temp);
        x.array[ii][j] = x.array[ii][j] * temp;
    }
    
    // now for every k not ii subtract row ii multiplied by xt[k][jj] from row k provided x[k][jj] != 0
    for(k=0;k<x.nr;++k)
    {
        if(k!=ii && x.array[k][jj].num!=0)
        {
            //temp.Re = x[k][jj].Re;
            //temp.Im = x[k][jj].Im;
            temp = x.array[k][jj];
            for(j=0;j<x.nc;++j)
            {
                //MulCC(temp1, x[ii][j], temp);
                //SubCC(x[k][j], x[k][j], temp1);
                temp1 = x.array[ii][j] * temp;
                x.array[k][j] = x.array[k][j] - temp1;
            }
        }
    }
    return true;
    
}/* doReduceModp */

void myFree(matq& x)
{
    INT32       i, j;
    
    if(x.array)
    {
        for(i=0;i<x.nr;++i)
        {
            for(j=0;j<x.nc;++j)
            {
                myFree(x.array[i][j]);
           }
            free(x.array[i]);
            x.array[i] = NULL;
        }
        free(x.array);
        x.array = NULL;
    }
    
}/* myFree */

void myFree(matq* z, int numMats)
{
    int     i;
    
    for(i=0;i<numMats;++i)
        myFree(z[i]);
    
}/* myFree */


// returns 1 if x=y else returns 0
INT32 matqCompare(const matq& x, const matq& y)
{
	INT32		i, j;
	
	if(x.nr!=y.nr || x.nc!=y.nc)
		return 0;
	
	for(i=0;i<x.nr;i++)
		for(j=0;j<x.nc;j++)
			if(x.array[i][j]!=y.array[i][j])
				return 0;
	
	return 1;
	
}/* matqCompare */


void add(matq& z, const matq& x, const matq& y)
{
	matq		zt;
	INT32		i, j;
	
	if(x.nr!=y.nr || x.nc!=y.nc)
		return;
    
    init(zt, x.nr, x.nc);
	
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j] = x.array[i][j] + y.array[i][j];
	
	z = zt;
	
}/* add */


void sub(matq& z, const matq& x, const matq& y)
{
	matq		zt;
	INT32		i, j;
	
	if(x.nr!=y.nr || x.nc!=y.nc)
		return;
	
	init(zt, x.nr, x.nc);
	
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j] = x.array[i][j] - y.array[i][j];
	
	z = zt;
    
}/* sub */


void mul(matq& z, const matq& x, const matq& y)
{
	matq		zt;
	INT32		i, j, k;
	
	if(x.nc!=y.nr)
		return;
	
	init(zt, x.nr, y.nc);
    
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
		{
			zt.array[i][j] = 0;
			for(k=0;k<x.nc;k++)
				zt.array[i][j] += x.array[i][k] * y.array[k][j];
		}
	
	z = zt;
	
}/* mul */
	

void mul(matq& z, const matq& x, const bf& y)
{
	matq		zt;
	INT32		i, j;
	
	init(zt, x.nr, x.nc);
    
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j] = y * x.array[i][j];
	
	z = zt;
	
}/* mul */

void mul(matq& z, const matq& x, const mb& y)
{
	matq		zt;
	INT32		i, j;
	
	init(zt, x.nr, x.nc);
	
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j] = y * x.array[i][j];
	
	z = zt;
	
}/* mul */


void mul(matq& z, const matq& x, double y)
{
	matq		zt;
	INT32		i, j;
	
	init(zt, x.nr, x.nc);
	
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j] = y * x.array[i][j];
	
	z = zt;
	
}/* mul */

void div(matq& z, const matq& x, const matq& y)
{
    matq    yInv;
    bf      det;
    
    inv(yInv, det, y);
    if(!det)
        return;
    
    z = x*yInv;
    
}/* div */


// creates unit matrix of size n
void matqUnit(matq& z, INT32 n)
{
    matq        zt;
	INT32		i, j;
    
    init(zt, n, n);
	
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
		{
			if(i==j)
				zt.array[i][j] = 1;
			else
				zt.array[i][j] = 0;
		}
    z = zt;
	
}/* matqUnit */


matq matqUnit(INT32 n)
{
	matq		z;
	
	matqUnit(z, n);
	
	return z;
	
}/* matqUnit */


void transpose(matq& z, const matq& x)
{
	matq		zt;
	INT32		i, j;
	
	init(zt, x.nc, x.nr);
	
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j] = x.array[j][i];
	
	z = zt;
	
}/* transpose */


matq transpose(const matq& x)
{
	matq		z;
	
	transpose(z, x);
	
	return z;
	
}/* transpose */

// x must be square
bool trace(bf& z, const matq& x)
{
	bf			zt;
	INT32		i;
	
	if(x.nr != x.nc)
	{
		z = 0;
		return false;
	}
	
	zt = 0;
	for(i=0;i<x.nr;i++)
		zt+=x.array[i][i];
	
	z = zt;
	
	return true;
	
}/* trace */


bf trace(const matq& x)
{
	bf		z;
	
	if(trace(z, x))
	{
		return z;
	}
	else
	{
		z = 0;
		return z;
	}

}/* trace */


// puts matrix y.array at bottom of x.array; the matrices must have the same number of columns
void matqRowAugment(matq& z, const matq& x, const matq& y)
{
	matq		zt;
	INT32		i, j;
	
	if(x.nc!=y.nc)
		return;
    
    init(zt, x.nr+y.nr, x.nc);
	
	// copy x.array into zt.array top
	for(i=0;i<x.nr;i++)
		for(j=0;j<x.nc;j++)
			zt.array[i][j] = x.array[i][j];
	
	// copy y.array to bottom of zt.array
	for(i=0;i<y.nr;i++)
		for(j=0;j<y.nc;j++)
			zt.array[i+x.nr][j] = y.array[i][j];
	
	z = zt;
	
}/* matqRowAugment */


matq matqRowAugment(const matq& x, const matq& y)
{
	matq	z;
	
	matqRowAugment(z, x, y);
	
	return z;
	
}/* matqRowAugment */


// puts matrix y.array to right of x.array; the matrices must have the same number of rows
void matqColAugment(matq& z, const matq& x, const matq& y)
{
	matq		zt;
	INT32		i, j;
	
	if(x.nr!=y.nr)
		return;
    
    init(zt, x.nr, x.nc+y.nc);
	
	zt.nr = x.nr;
	zt.nc = x.nc + y.nc;
	
	// copy x.array into zt.array left
	for(i=0;i<x.nr;i++)
		for(j=0;j<x.nc;j++)
			zt.array[i][j] = x.array[i][j];
	
	// copy y.array to right of zt.array
	for(i=0;i<y.nr;i++)
		for(j=0;j<y.nc;j++)
			zt.array[i][j+x.nc] = y.array[i][j];
	
	z = zt;
	
}/* matqColAugment */

matq matqColAugment(const matq& x, const matq& y)
{
	matq	z;
	
	matqColAugment(z, x, y);
	
	return z;
	
}/* matqColAugment */

// puts z.array in echelon form
void matqGaussElim(matq& z, INT32& swapParity, const matq& x)
{
    matq        xt;
    INT32       ii, jj;
    
    swapParity = 1;
    
    xt = x;
    
    ii = jj = 0;
    while(ii<xt.nr && jj<xt.nc)
    {
        if(doGauss(xt, swapParity, ii, jj))
        {
            ii++;
            jj++;
        }
        else
            jj++;
    }
    
    z = xt;
    
}/* matqGaussElim */


bool matqGaussElim(matq& z, INT32& swapParity, const char *xString)
{
	matq		xt;
	
	if(!matqConvFromStr(xt, xString))
		return false;
	
	matqGaussElim(z, swapParity, xt);
	
	return true;
	
}/* matqGaussElim */

// puts z.array in echelon form
// only for integer matrices
void matqGaussElimModq(matq& z, INT32& swapParity, const matq& x, INT32 q)
{
    matq        xt;
    INT32       ii, jj;
    
    swapParity = 1;
    
    xt = x % q;
    
    ii = jj = 0;
    while(ii<xt.nr && jj<xt.nc)
    {
        if(doGaussModq(xt, xt, swapParity, ii, jj, q))
        {
            ii++;
            jj++;
        }
        else
            jj++;
    }
    
    z = xt;
    
}/* matqGaussElimModq */


// by rows mod q
void matqGaussElim(matq& z, const matq& x, double q)
{
	matq	xt;
	INT32	i, j, k, i_max, ii;
	bool	inSearch;
	
	xt = x;
	ii = 0; // i -> pivot row; ii -> pivot column
	for(i=0;i<xt.nr-1;i++)
	{
		inSearch = true;
		while(inSearch)
		{
			// find row whose iith column is not zero at ith row and below
			i_max = i;
			for(k=i+1;k<xt.nr;k++)
			{
				if(abs(xt.array[k][ii])>abs(xt.array[i_max][ii]))
					i_max = k;
			}
			
			if(!xt.array[i_max][ii].num.n)
			{
				// iith column must be skipped since it's zero from ith row down
				ii++; // move to next column to right
				if(ii>=xt.nc)
				{
					z = xt;
					return;
				}
			}
			else
				inSearch = false;
		}
		
		if(!(i==i_max))
		{
			matqSwapRows(xt, xt, i, i_max);
		}
		
		// Eliminate the iith element of the rows below the ith
		
		for(k=i+1;k<xt.nr;k++)
		{
			for(j=ii+1;j<xt.nc;j++)
			{
				xt.array[k][j] = xt.array[i][ii] * xt.array[k][j] - xt.array[k][ii] * xt.array[i][j];
				xt.array[k][j].num = (xt.array[k][j].num % q);
			}
				
			xt.array[k][ii] = 0;
		}
		ii++;
		if(ii>=xt.nc)
		{
			z = xt;
			return;
		}
	}
	
	z = xt;
	
}/* matqGaussElim */

// reduced row echelon form; x must be in echolon form
void matqReduce(matq& z, const matq& x)
{
    matq            xt;
    INT32           ii, jj;
    
    init(xt, x.nr, x.nc);
   
    xt = x;
    
    ii = jj = 0;
    while(ii<xt.nr && jj<xt.nc)
    {
        if(doReduce(xt, ii, jj))
        {
            ii++;
            jj++;
        }
        else
            jj++;
    }
    z = xt;
    
}/* matqReduce */

void matqReduceModp(matq& z, const matq& x, double p)
{
    matq            xt;
    INT32           ii, jj;
    
    init(xt, x.nr, x.nc);
    
    xt = x;
    
    ii = jj = 0;
    while(ii<xt.nr && jj<xt.nc)
    {
        if(doReduceModp(xt, ii, jj, p))
        {
            ii++;
            jj++;
        }
        else
            jj++;
    }
    z = xt;
}

// https://en.wikipedia.org/wiki/Kernel_(linear_algebra)
void matqNullSpace(INT32& numNV, matq& nullVectors, const matq& x)
{
	matq		xt, yt, zt;
    matq        reduced;
	INT32		i, j, k;
	INT32		swapParity;
	
	xt = x;

	matqRowAugment(yt, xt, matqUnit(xt.nc));
	matqGaussElim(zt, swapParity, transpose(yt));
	matqReduce(reduced, zt);
    reduced = transpose(reduced); // x.nr+x.nc x x.nc
	
	// look for zero columns in reduced for row index from 0 to xt.nr; null vectors are at the bottems of these columns and have xt.nc elements
	// create a Boolean array to hold true or false for null vector columns
	bool	*hasNV;
	bool	isNull;
	hasNV = (bool*)malloc(xt.nc*sizeof(bool));
	for(j=0;j<xt.nc;j++)
	{
		isNull = true;
		for(i=0;i<xt.nr;i++)
		{
			if(reduced.array[i][j].num.n)
			{
				isNull = false;
				break;
			}
		}
		if(isNull)
			hasNV[j] = true;
		else
			hasNV[j] = false;
	}
	
	// get number of null vectors
	numNV = 0;
	for(i=0;i<xt.nc;i++)
	{
		if(hasNV[i])
			numNV++;
	}
	
	if(!numNV)
	{
		free(hasNV);
        init(nullVectors, 1, 1);
		nullVectors.array[0][0] = 0;
		return;
	}
	
	// now have null vectors which we want as columns of myNullVectors.array
	nullVectors.nr = xt.nc;
	nullVectors.nc = numNV;  // column size of xt
    init(nullVectors, nullVectors.nr, nullVectors.nc);
	
	// copy null vectors into columns of nullVectors.array and get lcm of den's and multiply after that
	k = 0;
	for(j=0;j<xt.nc;j++)
	{
		if(hasNV[j])
		{
			for(i=0;i<nullVectors.nr;i++)
				nullVectors.array[i][k] = reduced.array[xt.nr+i][j];
			k++;
		}
	}
	
    // get lcm for each column and multiply that column
	mb		myLCM;
	
	for(j=0;j<nullVectors.nc;j++)
	{
		myLCM = 1;
		for(i=0;i<nullVectors.nr;i++)
			myLCM = lcm(myLCM, nullVectors.array[i][j].den);
		if(!(myLCM==1))
           for(i=0;i<nullVectors.nr;i++)
               nullVectors.array[i][j] = myLCM*nullVectors.array[i][j];
	}
	
	free(hasNV);

}/* matqNullSpace */


bool matqNullSpace(INT32& numNV, matq& nullVectors, const char *xString)
{
	matq		x;
	
	if(!matqConvFromStr(x, xString))
		return false;
	
	matqNullSpace(numNV, nullVectors, x);
	
	return true;
	
}/* matqNullSpace */

// https://en.wikipedia.org/wiki/Kernel_(linear_algebra)
// only for integer matrices
void matqNullSpaceModq(INT32& numNV, matq& nullVectors, const matq& x, INT32 q)
{
    matq        xt, yt, zt;
    matq        reduced;
    INT32        i, j, k;
    INT32        swapParity;
    poly         temp;
    
    xt = x % q;
    
    matqRowAugment(yt, xt, matqUnit(xt.nc));
    matqGaussElimModq(zt, swapParity, transpose(yt), q);
    matqReduceModp(reduced, zt, q);
    reduced = transpose(reduced); // x.nr+x.nc x x.nc
    /*
    for(i=0;i<reduced.nr;++i)
        for(j=0;j<reduced.nc;++j)
        {
            cout << reduced.array[i][j] << "  ";
            if(j==reduced.nc-1)
                cout << endl;
        }
    */
    // look for zero columns in reduced for row index from 0 to xt.nr; null vectors are at the bottems of these columns and have xt.nc elements
    // create a Boolean array to hold true or false for null vector columns
    bool    *hasNV;
    bool    isNull;
    hasNV = (bool*)malloc(xt.nc*sizeof(bool));
    for(j=0;j<xt.nc;j++)
    {
        isNull = true;
        for(i=0;i<xt.nr;i++)
        {
            if(reduced.array[i][j].num.n)
            {
                isNull = false;
                break;
            }
        }
        if(isNull)
            hasNV[j] = true;
        else
            hasNV[j] = false;
    }
    
    // get number of null vectors
    numNV = 0;
    for(i=0;i<xt.nc;i++)
    {
        if(hasNV[i])
            numNV++;
    }
    
    if(!numNV)
    {
        free(hasNV);
        init(nullVectors, 1, 1);
        nullVectors.array[0][0] = 0;
        return;
    }
    
    // now have null vectors which we want as columns of myNullVectors.array
    nullVectors.nr = xt.nc;
    nullVectors.nc = numNV;  // column size of xt
    init(nullVectors, nullVectors.nr, nullVectors.nc);
    
    // copy null vectors into columns of nullVectors.array and get lcm of den's and multiply after that
    k = 0;
    for(j=0;j<xt.nc;j++)
    {
        if(hasNV[j])
        {
            for(i=0;i<nullVectors.nr;i++)
                nullVectors.array[i][k] = reduced.array[xt.nr+i][j];
            k++;
        }
    }
    /*
    cout << "nullVectors = " << endl;
    for(i=0;i<nullVectors.nr;++i)
        for(j=0;j<nullVectors.nc;++j)
        {
            cout << nullVectors.array[i][j] << "  ";
            if(j==nullVectors.nc-1)
                cout << endl;
        }
     */
    
    // get lcm for each column and multiply that column
    mb        myLCM;
    
    for(j=0;j<nullVectors.nc;j++)
    {
        myLCM = 1;
        for(i=0;i<nullVectors.nr;i++)
            myLCM = lcm(myLCM, nullVectors.array[i][j].den);
        if(!(myLCM==1))
            for(i=0;i<nullVectors.nr;i++)
                nullVectors.array[i][j] = myLCM*nullVectors.array[i][j];
    }
    /*
    cout << "nullVectors = " << endl;
    for(i=0;i<nullVectors.nr;++i)
        for(j=0;j<nullVectors.nc;++j)
        {
            cout << nullVectors.array[i][j] << "  ";
            if(j==nullVectors.nc-1)
                cout << endl;
        }
    */
    if(numNV==1)
    {
        // if nullVectors[0] is just a constant, we really have no null vectors
        init(temp, nullVectors.nr-1);
        for(i=0;i<=temp.deg;++i)
            temp.array[i] = nullVectors.array[i][0];
        temp = polyNormalize(temp);
        if(!temp.deg)
        {
            numNV = 0;
            free(hasNV);
            return;
        }
    }
    
    free(hasNV);
    
}/* matqNullSpaceModq */


bool matqInvert(matq& z, bf& det, const matq& x)
{
	matq		xt, zt;
	INT32		i, j;
	INT32		swapParity;
	
	if(x.nr!=x.nc)
    {
        det = 0;
        return false;
    }
		
	
	if(x.nr==1)
	{
		zt = x;
		if(!zt.array[0][0].num.n)
			return false;
		
		zt.array[0][0] = 1 / zt.array[0][0];
        z = zt;
        return true;
	}
	
	xt = matqColAugment(x, matqUnit(x.nr));
	
	matqGaussElim(xt, swapParity, xt);
	det = swapParity;
	for(i=0;i<x.nr;i++)
		det = det * xt.array[i][i];
	
    if(!det.num)
    {
        return false;
    }
    
	matqReduce(xt, xt);
	
    init(zt, x.nr, x.nc);
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j] = xt.array[i][j+zt.nc];
	
    z = zt;
	return true;
	
}/* matqInvert */

matq matqInvert(bf& det, const matq& x)
{
	matq		z;
	
	if(!matqInvert(z, det, x))
		z = x;
	
	return z;
	
}/* matqInvert */

// only for integer matrices
bool matqInvertModp(matq& z, bf& det, const matq& x, double p)
{
    matq        xt, zt;
    INT32       i, j;
    INT32       swapParity;
    bf          temp;
    
    if(x.nr!=x.nc)
    {
        det = 0;
        return false;
    }
    
    
    if(x.nr==1)
    {
        zt = x % p;
        if(!zt.array[0][0].num.n)
        {
            z = zt;
            return false;
        }
        temp = xt.array[0][0];
        temp.num = invMod(temp.num, p);
        zt.array[0][0] = temp;
        z = zt;
        return true;
    }
    
    xt = matqColAugment(x, matqUnit(x.nr));
    
    matqGaussElimModq(xt, swapParity, xt, p);
    det = swapParity;
    for(i=0;i<x.nr;i++)
        det = det * xt.array[i][i];
    det.num = det.num % p;
    det = det.num;
    if(!det.num)
        return false;
    
    matqReduceModp(xt, xt, p);
    xt = xt % p;
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = xt.array[i][j+zt.nc];
    
    z = zt;
    return true;
    
}/* matqInvertModp */

matq matqInvertModp(bf& det, const matq& x, double p)
{
    matq    z;
    
    matqInvertModp(z, det, x, p);
    
    return z;
    
}/* matqInvertModp */

// meant for integer matrices
void matqMod(matq& z, const matq& x, const mb& q)
{
    matq        zt;
	INT32		i, j;
	
	zt = x;
	
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j].num = zt.array[i][j].num % q;
    z = zt;
	
}/* matqMod */

// meant for integer matrices
void matqMod(matq& z, const matq& x, double q)
{
    matq        zt;
	INT32		i, j;
	
	zt = x;
	
	for(i=0;i<zt.nr;i++)
		for(j=0;j<zt.nc;j++)
			zt.array[i][j].num = zt.array[i][j].num % q;
    
    z = zt;

}/* matqMod */


// z = x^y ; x must be a square matrix and y>=0
bool power(matq& z, const matq& x, INT32 y)
{
	INT32				i, n, sy;
	static UINT32		mask; // 0x80000000
	//static mb			one;
	matq				zt;
	static bool			initGood=false;
	
	if(!initGood)
	{
		//init(one, 1);
		mask = 1;
		mask = (mask<<31); // 0x80000000
		initGood = true;
	}
	
	if(x.nr!=x.nc || y<0)
	{
		//z = 0;
		return false;
	}
		
	if(matqCompare(x, matqUnit(x.nr)))
	{
		z = matqUnit(x.nr);
		return true;
	}
	
	if(y==0)
	{
		z = matqUnit(x.nr);
		return true;
	}
	
	zt = matqUnit(x.nr);
	sy = y;
	n = NumBits(sy);
	
	sy = (sy<<(32-n));  // go to leading 1 bit
	
	for(i=0;i<n;i++)
	{
		mul(zt, zt, zt); // zt^2
		if(sy & mask)
			mul(zt, zt, x);
		sy = (sy<<1);
	}
	
	z = zt;
	
	return true;
	
}/* power */


// z = x^y mod q ; x must be a square matrix and y>=0
bool powerMod(matq& z, const matq& x, const mb& y, const mb& q)
{
	INT32				i, n;
	mb					mask;
	matq				zt;
	
	if(x.nr!=x.nc || y.n<0)
	{
		//z = 0;
		return false;
	}
	
	if(matqCompare(x, matqUnit(x.nr)))
	{
		z = matqUnit(x.nr);
		return true;
	}
	
	if(y.n==0)
	{
		z = matqUnit(x.nr);
		return true;
	}
	
	zt = matqUnit(x.nr);
    //cout << "zt = " << zt << endl;
	n = NumBits(y);
	
	// need to make mask have n bits with leading bit one and all others zero
	mask = 1;
	mask = (mask<<(n-1));
	
	for(i=0;i<n;i++)
	{
		mul(zt, zt, zt); // zt^2
		zt = zt % q;
		if((y & mask).n)
		{
			mul(zt, zt, x);
			zt = zt % q;
		}
		mask = (mask>>1);
	}
	//cout << "zt = " << zt << endl;
	z = zt;
	
	return true;
	
}/* powerMod */

// creates permutation matrix which, x from left, interchanges row1 with row2
static matq createPermutationMatrix(INT32 n, INT32 row1, INT32 row2)
{
    matq    P;
    
    if (n<1)
    {
        n = 1;
    }
    
    P = matqUnit(n);
    if (row1==row2 || row1<0 || row2<0 || row1>=n || row2>=n)
    {
        return P;
    }
    
    P.array[row1][row1] = 0;
    P.array[row2][row2] = 0;
    P.array[row1][row2] = 1;
    P.array[row2][row1] = 1;
    return P;
    
}/* createPermutationMatrix */

// adds transpose permutation Q to P, returns Q
static matq addTranspose(matq& P, INT32 row1,  INT32 row2, INT32& swapParity)
{
    INT32   n;
    matq    Q;
    
    n = P.nc;
    if (row1==row2 || row1<0 || row2<0 || row1>=n || row2>=n)
    {
        Q = matqUnit(n);
        return Q;
    }
    
    Q = createPermutationMatrix(n, row1, row2);
    
    P = Q * P;
    swapParity = -swapParity;
    
    return Q;
    
}/* addTranspose */

// if A has zeroes on diagonal, PA does not
static bool permutationMatrix(const matq& A, matq& P, INT32& swapParity)
{
    INT32   n, i, j, k;
    matq    B, Q;
    bool    isZero;
    bf      temp;
    INT32   count, countMax=362880; // 10! / 10
    
    swapParity = 1;
    
    n = A.nc;
    P = matqUnit(n);
    
    if (n!=A.nr)
    {
        return false;
    }
    
    isZero = false;
    for (i=0; i<n; ++i)
    {
        if (A.array[i][i].num==0)
        {
            isZero = true;
            break;
        }
    }
    if (!isZero)
    {
        return true; // P is unit matrix
    }
    
    
    B = A;
    isZero = true;
    count = 0;
    while(isZero && count<countMax)
    {
        count++;
        
        for (j=0; j<n; ++j)
        {
            if (B.array[j][j]==0)
            {
                temp = 0;
                for (i=0; i<n; ++i)
                {
                    if (abs(B.array[i][j])>temp)
                    {
                        temp = abs(B.array[i][j]);
                        k = i;
                    }
                }
                if (temp==0)
                {
                    return false;
                }
                Q = addTranspose(P, j, k, swapParity);
                B = Q * B;
            }
        }
        // some A's will still give a B with a zero on diagonal
        isZero = false;
        for (i=0; i<n; ++i)
        {
            if (B.array[i][i].num==0)
            {
                isZero = true;
                break;
            }
        }
    }
    
    if(isZero)
        return false;
    
    return true;
    
}/* permutationMatrix */

// Doolittle algorithm, L is unit lower diagonal; A = P L U
bool LUdecomp(const matq& A, matq& P, matq& L, matq& U, INT32& swapParity)
{
    INT32   n, i, j, k;
    bf      alpha;
    
    if(!matqDet(A))
        return false;
    
    if(!permutationMatrix(A, P, swapParity))
    {
        return false;
    }
    
    U = P * A;
    
    n = U.nr;
    for (i=0; i<n; ++i)
    {
        for (j=0; j<i; ++j)
        {
            alpha = U.array[i][j];
            for (k=0; k<j; ++k)
            {
                alpha = alpha - U.array[i][k] * U.array[k][j];
            }
            U.array[i][j] = alpha / U.array[j][j];
        }
        
        for (j=i; j<n; ++j)
        {
            alpha = U.array[i][j];
            for (k=0; k<i; ++k)
            {
                alpha = alpha - U.array[i][k] * U.array[k][j];
            }
            U.array[i][j] = alpha;
        }
    }
    
    // I -> L; modify U
    L = matqUnit(n);
    for (j=0; j<n; ++j)
    {
        for (i=j+1; i<n; ++i)
        {
            L.array[i][j] = U.array[i][j];
            U.array[i][j] = 0;
        }
    }
    
    P = transpose(P);
    
    return true;
    
}/* LUdecomp */


bf matqDet(const matq& A)
{
    bf      det;
    
    matqInvert(det, A);
    
    return det;
    
}/* matqDet */


// z = inverse(x)
void inv(matq& z, bf& determinant, const matq& x)
{
    matq            xt;
    INT32           n, i, j, k, L, *ik, *jk;
    bf              amax, save, zero=0., one = 1.;
    
    if(x.nr!=x.nc)
    {
        determinant = 0;
        return;
    }
    
    n = x.nr;
    
    init(xt, x.nr, x.nc);
    xt = x;
    
    ik = (INT32 *)malloc(n*sizeof(INT32));
    jk = (INT32 *)malloc(n*sizeof(INT32));
    
    determinant = one;
    
    for(k=0; k<n; k++)
    {
        /* find largest element in abs value in rest of matrix */
        amax = zero;
        
        for(i=k; i<n; i++)
        {
            for(j=k; j<n; j++)
            {
                if(abs(xt.array[i][j]) > abs(amax))
                {
                    amax = xt.array[i][j];
                    ik[k] = i;
                    jk[k] = j;
                }
            }
        }
        
        if(!abs(amax))
        {
            z.nr = z.nc = 0;
            z.array = NULL;
            determinant.num = 0;
            determinant.den = 1;
            return;
        }
        
        /* interchange rows and columns to put amax in array(k,k) */
        i = ik[k];
        if(i>k)
        {
            for(j=0; j<n; j++)
            {
                save = xt.array[k][j];
                xt.array[k][j] = xt.array[i][j];
                xt.array[i][j] = -save;
            }
        }
        
        j = jk[k];
        if(j>k)
        {
            for(i=0; i<n; i++)
            {
                save = xt.array[i][k];
                xt.array[i][k] = xt.array[i][j];
                xt.array[i][j] = -save;
            }
        }
        
        /* accumulate elements of inverse matrix */
        for(i=0; i<n; i++)
        {
            if(i!=k)
                xt.array[i][k] = -xt.array[i][k]/amax;
        }
        
        for(i=0; i<n; i++)
        {
            for(j=0; j<n; j++)
            {
                if(i!=k && j!=k)
                    xt.array[i][j] = xt.array[i][j] + xt.array[i][k]*xt.array[k][j];
            }
        }
        
        for(j=0; j<n; j++)
        {
            if(j!=k)
                xt.array[k][j] = xt.array[k][j]/amax;
        }
        
        xt.array[k][k] = one/amax;
        determinant = determinant*amax;
    }
    
    /* restore ordering of the matrix */
    for(L=0; L<n; L++)
    {
        k = n - L - 1;
        j = ik[k];
        
        if(j>k)
        {
            for(i=0; i<n; i++)
            {
                save = xt.array[i][k];
                xt.array[i][k] = -xt.array[i][j];
                xt.array[i][j] = save;
            }
        }
        
        i = jk[k];
        
        if(i>k)
        {
            for(j=0; j<n; j++)
            {
                save = xt.array[k][j];
                xt.array[k][j] = -xt.array[i][j];
                xt.array[i][j] = save;
            }
        }
    }
    
    z = xt;
    
    free(ik);
    free(jk);
    
} /* inv */


matq inv(const matq& x)
{
    bf     det;
    matq   z;
    inv(z, det, x);
    
    return z;
    
}/* inv */


bf determinant(const matq& x)
{
    matq    z;
    bf      det;
    
    inv(z, det, x);
    
    return det;
    
}/* determinant */

void genRandIntMatOverp(matq& z, int nr, int nc, double p)
{
    matq    zt;
    int     i, j;
    int     pInt;
    
    if(p<2)
        return;
    
    pInt = p;
    
    
    init(zt, nr, nc);
    
    for(i=0;i<zt.nr;++i)
        for(j=0;j<zt.nc;++j)
        {
            if(pInt>2)
                zt.array[i][j] = rand() % pInt - pInt/2;
            else
                zt.array[i][j] = rand() % pInt;
        }
    
    z = zt;
    
}/* genRandIntMatOverp */

matq genRandIntMatOverp(int nr, int nc, double p)
{
    matq    z;
    
    genRandIntMatOverp(z, nr, nc, p);
    
    return z;
    
}/* genRandIntMatOverp */

// checks if matrix is diagonal
bool isDiag(const matq& x)
{
    int     i, j;
    bool    isDiag;
    
    if(x.nr!=x.nc)
        return false;
    
    isDiag = true;
    for(i=0;i<x.nr;++i)
        for(j=0;j<x.nc;++j)
            if(i!=j)
            {
                if(!(x.array[i][j].num==0))
                {
                    isDiag = false;
                    break;
                }
            }
    return isDiag;
    
}/* isDiag */

bf genMatrixSeed(matq& seed, int m, double p)
{
    int     i, k;
    matq    a, I, testI, zt, ztInv;
    mb      pmb, kmb;
    bool    isGroup, isI;
    bf      det;
    
    I = matqUnit(m);
    k = pow(p, m) - 1; // want a^k = I
    equate(pmb, p);
    isGroup = false;
    while(!isGroup)
    {
        equate(kmb, k);
        zt = I;
        det = 0;
        while(!det.num)
        {
            myFree(zt);
            genRandIntMatOverp(zt, m, m, p);
            matqInvertModp(ztInv, det, zt, p);
            myFree(ztInv);
        }
        // zt is candidate
        if(isDiag(zt))
            continue;
        // check if zt^k = I mod p
        powerMod(testI, zt, kmb, pmb);
        if(!(testI==I))
        {
            myFree(testI);
            continue;
        }
       // cout << "zt = " << zt << endl;
        // have possible seed matrix zt
        // now want to check for i=2, 3, ... k/2 & if i | k whether zt^i = I mod p, if so start over with new zt
        isI = false;
        for(i=2;i<=k/2;++i)
        {
            if(i*(k/i)==k)
            {
                myFree(testI);
                equate(kmb, i);
                powerMod(testI, zt, kmb, pmb);
                if(testI==I)
                {
                    //cout << "testI = " << testI << endl;
                    isI = true;
                    break;
                }
            }
        }
        if(isI)
            continue;
        isGroup = true;
    }
    
    seed = zt;
    return det;
    
}/* genMatrixSeed */

// generates matrix group of order numMats = p^m - 1 where m is size of the square matrices
void genMatrixGroup(matq*& z, int& numMats, int m, int p)
{
    matq    x, zt, I, ztInv;
    int     i;
    int     order, orderTarget;
    bf      det;
    
    orderTarget = pow(p, m) - 1;
    I = matqUnit(m);
    order = 0;
    while(order != orderTarget)
    {
        zt = I;
        det = 0;
        while(!det.num)
        {
            myFree(zt);
            genRandIntMatOverp(zt, m, m, p);
            matqInvertModp(ztInv, det, zt, p);
            myFree(ztInv);
        }
        x = zt;
        order = 1;
        while(!(x==I))
        {
            order++;
            x = x * zt % p;
        }
    }
    
    // now have the group
    z = (matq*)malloc(order*sizeof(matq));
    init(z, m, m, order);
    numMats = order;
    x = zt;
    z[0] = I;
    z[1] = zt;
    for(i=2;i<numMats;++i)
    {
        x = x * zt % p;
        z[i] = x;
    }
    
}/* genMatrixGroup */

// if field, table is for addition operation
// https://en.wikipedia.org/wiki/Finite_field#Applications
// Zech's Logarithms  a^j + a^i = a^i (a^(j-i) + I) ; so just need to verify a^n + I, n = 0, 1, ... , numMats-1
// belongs to field
bool isGroupField(matq& table, const matq* x, int numMats, int p)
{
    int     i, j, k, m, n;
    matq    table1, sum, temp;
    int     *zechLog;
    bool    isThere, isZero;
    
    if(!numMats)
        return false;
    
    m = x[0].nr; // size of square matrices
    
    k = numMats;
    
    zechLog = (int*)malloc(k*sizeof(int));
    
    init(table1, k, k);
    
    // load zechLog
    for(i=0;i<k;++i)
    {
        temp = (x[i] + x[0]) % p;
        isThere = false;
        for(j=0;j<k;++j)
        {
            if(!temp)
            {
                zechLog[i] = -1;
                isThere = true;
                break;
            }
            if(temp==x[j])
            {
                zechLog[i] = j;
                isThere = true;
                break;
            }
        }
        if(!isThere)
            return false;
    }
   // for(i=0;i<numMats;++i)
    //    cout << zechLog[i] << endl;
    
    // now we need to add using Zech a^j + a^i = a^i (a^(j-i) + I)
    for(i=0;i<k;++i)
    {
        for(j=i;j<k;++j)
        {
            isZero = false;
            if(zechLog[j-i]==-1)
            {
                table1.array[i][j] = -1;
                table1.array[j][i] = -1;
                isZero = true;
            }
            if(!isZero)
            {
                n = (i + zechLog[j-i]) % numMats;
                if(n<0)
                    n+=k;
                //cout << "n = " << n << endl;
                table1.array[i][j] = n;
                table1.array[j][i] = n;
            }
        }
    }
    
    table = table1;
    free(zechLog);
    return true;
    
}/* isGroupField */

void countMatrixGroups(int& countkm, int& countLess, int& countMore, int numTrials, int m, double p)
{
    matq    a, zt, ztInv, I;
    int     i, k, km;
    bf      det;
    bool    isGood;
    
    km = pow(p, m) - 1;
    I = matqUnit(m);
    countLess = 0;
    countkm = 0;
    countMore = 0;
    for(i=0;i<numTrials;++i)
    {
        isGood = false;
        while(!isGood)
        {
            zt = genRandIntMatOverp(m, m, p);
            isGood = matqInvertModp(ztInv, det, zt, p);
            if(isGood)
                myFree(ztInv);
            if(isGood && isDiag(zt))
                isGood = false;
            if(!isGood)
                myFree(zt);
        }
        // cout << "zt = " << zt << endl;
        a = zt;
        // form group
        k = 1;
        while(!(a==I))
        {
            a = (a * zt) % p;
            k++;
        }
        
        if(k<km)
            countLess++;
        if(k==km)
            countkm++;
        if(k>km)
            countMore++;
    }
    
}/* countMatrixGroups */

int comparInt(const void* a, const void* b)
{
    if ( *(int*)a <  *(int*)b ) return -1;
    if ( *(int*)a == *(int*)b ) return 0;
    return 1;
    
}/* comparInt */

void findMatrixGroupOrders(int*& order, int*& freq, int& numOrders, int numTrials, int m, double p)
{
    matq    a, zt, ztInv, I;
    int     i, j, k;
    bf      det;
    bool    isGood;
    int     *ordert, order1;
    
    I = matqUnit(m);
    ordert = (int*)malloc(numTrials*sizeof(int));
    
    for(i=0;i<numTrials;++i)
    {
        isGood = false;
        while(!isGood)
        {
            zt = genRandIntMatOverp(m, m, p);
            isGood = matqInvertModp(ztInv, det, zt, p);
            if(isGood)
                myFree(ztInv);
            if(!isGood)
                myFree(zt);
        }
        a = zt;
        // form group
        k = 1;
        while(!(a==I))
        {
            a = (a * zt) % p;
            k++;
        }
        ordert[i] = k;
    }
    qsort(ordert, numTrials, sizeof(int), comparInt);
    
    // we need to count the number of different orders and put that in numOrders
    numOrders = 0;
    order1 = 0;
    for(i=0;i<numTrials;++i)
    {
        if(order1 != ordert[i])
        {
            numOrders++;
            order1 = ordert[i];
        }
    }
    order = (int*)malloc(numOrders*sizeof(int));
    freq = (int*)malloc(numOrders*sizeof(int));
    for(i=0;i<numOrders;++i)
        freq[i] = 0;
    
    // now need to fill the order and freq arrays
    j = 0; // index to order and freq arrays
    order1 = ordert[j];
    order[j] = ordert[j];
    freq[j]++; // there is at least one of that order
    for(i=1;i<numTrials;++i)
    {
        if(order1 == ordert[i])
            freq[j]++;
        
        if(order1 != ordert[i])
        {
            order[++j] = ordert[i];
            freq[j]++;
            order1 = ordert[i];
        }
    }
    
    free(ordert);
    
}/* findMatrixGroupOrders */

// gives number of dimension m invertible matrices over GF(p), p prime, as
// (p^m - p^0) (p^m - p) (p^m - p^2) ... (p^m - p^(m-1))
void numInverts(mb& z, int m, mb& p)
{
    mb      term, term1;
    int     i;
    
    term = power(p, m);
    term1 = 1;
    z = 1;
    
    for(i=0;i<m;++i)
    {
        z = z * (term - term1);
        term1*=p;
    }
    
}/* numInverts */

mb numInverts(int m, mb& p)
{
    mb  z;
    
    numInverts(z, m, p);
    
    return z;
    
}/* numInverts */

// generates matrix cyclic group of order p^m - 1 from input square matrix of size m
bool genMatrixCyclicGroup(matq*& z, const matq x, double p)
{
    int     i, m, order;
    matq    temp, I;
    
    if(x.nr!=x.nc)
        return false;
    
    m = x.nr;
    order = pow(p, m) - 1;
    
    I = matqUnit(x.nr);
    init(z, m, m, order);
    
    z[0] = I;
    z[1] = x;
    temp = x;
    for(i=2;i<order;++i)
    {
        temp = temp * x % p;
        z[i] = temp;
    }
    temp = temp * x % p;
    // check that temp=I
    if(!(temp==I))
        return false;
    return true;
    
}/* genMatrixCyclicGroup */

// https://en.wikipedia.org/wiki/Finite_field#Applications
// Zech's Logarithms  a^j + a^i = a^i (a^(j-i) + I) ; so just need to verify a^n + I, n = 0, 1, ... , order-1   order = p^m - 1
// belongs to field
// this generates Zech's Logarithms for cyclic matrix group based on seed matrix x
// when poly for log is zero, we put -1 for log since zero means x^0 (x here is not seed matrix!)
bool matrixZechLogs(int*& z, const matq& x, double p)
{
    matq    *mGroup, temp, zero;
    int     i, j, m, order;
    bool    isThere;
    
    if(x.nr!=x.nc)
        return false;
    
    if(!genMatrixCyclicGroup(mGroup, x, p))
        return false;
    
    m = x.nr;
    order = pow(p, m) - 1;
    z = (int*)malloc(order*sizeof(int));
    
    init(zero, m, m);
    for(i=0;i<m;++i)
        for(j=0;j<m;++j)
            zero.array[i][j] = 0;
    
    // load zechLog
    for(i=0;i<order;++i)
    {
        temp = (mGroup[i] + mGroup[0]) % p;
        isThere = false;
        for(j=0;j<order;++j)
        {
            if(temp==zero)
            {
                z[i] = -1;
                isThere = true;
                break;
            }
            if(temp==mGroup[j])
            {
                z[i] = j;
                isThere = true;
                break;
            }
        }
        if(!isThere)
            return false;
    }
    
    return true;
    
}/* matrixZechLogs */


/*
 Finds the characteristic polynomial of a square matrix using a method described by
 Prof. Bumby of Rutgers Univ.
 
 Let M be the matrix of size nxn, and the characteristic polynomial be:
 
 x^n - p[1]*x^n-1 - p[2]*x^n-2 - ... - p[n], and let p[0] = 1
 
 Then define matrices F such that
 
 F_1(M) = M
 
 and for k = 1, 2,...,n
 
 p[k] = (1/k)*Tr(F_k)
 
 F_k+1 = M*(F_k - p[k]*I)
 */
bool charPoly(poly& z, const matq& x)
{
    poly    zt;
    matq    F, I;
    int     k, m; // size of x matrix and degree of poly z
    
    if(x.nr!=x.nc || !x.nr)
        return false;
    
    m = x.nr;
    init(zt, m);
    zt.array[zt.deg] = 1;
    
    I = matqUnit(m);
    
    // -p[k] <-> zt.array[zt.deg-k]
    F = x;
    for(k=1;k<=m;++k)
    {
        zt.array[zt.deg-k] = -trace(F) / k;
        F = x*(F + I*zt.array[zt.deg-k]);
    }
    z = zt;
    return true;
    
}/* charPoly */

static bool isRowZero(const matq& x, long row)
{
    bool    isZero;
    long    j;
    
    isZero = true;
    for(j=0;j<x.nc;j++)
    {
        if(x.array[row][j]!=0)
        {
            isZero = false;
            break;
        }
    }
    return isZero;

}/* isRowZero */

long matqRank(const matq& x)
{
    long    rank, i;
    matq    z, xt;
    int     swapParity;
    
    rank = 0;
    if(!x.nr)
        return rank;
    
    if(x.nr==1)
    {
        if(isRowZero(x, x.nr))
            return 0;
        else
            return 1;
    }
    matqGaussElim(xt, swapParity, x);
    matqReduce(z, xt);
    
    for(i=0;i<z.nr;++i)
    {
        if(!isRowZero(z, i))
            rank++;
    }
    return rank;
    
}/* matqRank */


void matqEigenvectors(matc& eigenvectors, matc& eigenvalues, long*& algMultiplicity, long*& geoMultiplicity, const matq& x, long myDecPrec, const fp& zeroSet)
{
    matc        xc;
    fp          temp;
    long        i, j;
    
    if(x.nr!=x.nc || x.nr<1)
        return;
    
    init(xc, x.nr, x.nc);
    for(i=0;i<x.nr;++i)
        for(j=0;j<x.nc;++j)
        {
            temp = x.array[i][j].num;
            if(x.array[i][j].den==1)
                xc.array[i][j] = temp;
            else
                xc.array[i][j] = temp / x.array[i][j].den;
        }
    
    matcEigenvectors(eigenvectors, eigenvalues, algMultiplicity, geoMultiplicity, xc, myDecPrec, zeroSet);
    //cout << eigenvalues << endl;
    
}/* matqEigenvectors */
