//
//  myGammaFunctions.cpp
//  fp64
//
//  Created by Bob Delaney on 2/1/19.
//  Copyright © 2019 Bob Delaney. All rights reserved.
//

#include "myGammaFunctions.hpp"
#include <time.h>

extern long blockPrec, decPrec;

// Gamma(x+1) is given by
// pow(x+a, (x+.5))*exp(-(x+a)) times
// Spouge formula sum from Wikipedia
// Sqrt(2*Pi) + Sum(k=1 to N) c(k)/(x+k)
// where N = a - 1 and
// c(0) = sqrt(2*Pi)
// c(k) = (-1)^(k-1) (a-k)^(k-.5) e^(a-k) / (k-1)!  k = 1. 2. ... , a-1
void GammaSum(fp& z, const fp& x, long a)
{
    long            k;
    fp              zt, ck, den, temp, myE;
    //clock_t         time;
    
    exp(myE, 1.);
    
    // c(1)
    temp = a - 1;
    den = exp(-temp);
    zt = sqrt(2*Pi());
    zt = zt + sqrt(temp)/((x+1.)*den);
    for(k=2;k<a;k++)
    {
        den = den*(k-1)*myE;
        temp = a - k;
        ck = pow(temp, k-.5)/den;
        if(k==2*(k/2))
            ck = -ck;
        zt = zt + ck/(x+k);
    }
    
    z = zt;
    
}/* GammaSum */

static fp GammaSum(const fp& x, long a)
{
    fp        z;
    
    GammaSum(z, x, a);
    
    return z;
    
}/* GammaSum, */

// uses Spouge formula for Gamma(x+1)
// thus this only holds for x>=0
void GammaSpouge(fp& z, const fp& x, long a)
{
    fp              zt, xt, afp;
    long            p;
    //clock_t         time;
    
    xt = x;
    p = getDecPrec();
    //a = 60; // 60
    //a = (5*(p+16))/4 - 10;
    zt = GammaSum(xt, a);
    zt = zt*pow(xt+a, (xt+.5))*exp(-xt-a);
    z = zt;
    
}/* GammaSpouge */

// use reflection formula for negative arguments?
// for x+n>1 use Gamma(x+n) = x(x+1)(x+2)...(x+n-1)Gamma(x) to reduce to x<1
// if x is negative find integer n for which x+n is first positive
// then Gamma(x) = Gamma(x+n)/[x(x+1)(x+2)...(x+n-1)]
// use smallIncompleteGamma and largeIncompleteGamma to calculate the Gamma(x) or Gamma(x+n)
// uses first formula of 6.5.12 in "Handbook of Mathematical Functions"
void smallIncompleteGamma(fp& z, const fp& x, const fp& a)
{
    fp                  s, s1, t, t1, xinc;
    long                i;
    long                p;
    
    if(!a.i.s || !x.i.s)
    {
        z = 0;
        return;
    }
    
    p = getBlockPrec();
    xinc = x;
    s = 0;
    s1 = 0;
    div(t, 1, x);
    
    for (i=1; ; i++)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t1, t, a);
        add(xinc, xinc, 1);
        div(t, t1, xinc);
    }
    
    //res = s*exp(-a + x*log(a));
    log(s1, a);
    mul(s1, s1, x);
    sub(s1, s1, a);
    exp(s1, s1);
    mul(z, s, s1);
    
}/* smallIncompleteGamma */

// Essentially this uses the repeated fraction formula 6.5.31 in "Handbook of Mathematical Functions"
// The recursion relation in this routine carries out the repeated fraction.
void largeIncompleteGamma(fp& z, const fp& x, const fp& a)
{
    fp              a0, a1, b0, b1, s, s1, s2;
    long            n;
    //long            p;
    
    if(!a.i.s)
    {
        z = 0;
        return;
    }
    
    a0 = 0;
    b0 = 1.;
    a1 = 1.;
    b1 = a;
    div(s, a1, b1);
    
    for(n=1;;n++)
    {
        //a0 = a1 + (n - x)*a0;
        sub (s1, n, x);
        mul(s1, s1, a0);
        add(a0, s1, a1);
        
        //b0 = b1 + (n - x)*b0;
        sub (s1, n, x);
        mul(s1, s1, b0);
        add(b0, s1, b1);
        
        //a1 = a*a0 + n*a1;
        mul(s1, n, a1);
        mul(s2, a, a0);
        add(a1, s1, s2);
        
        //b1 = a*b0 + n*b1;
        mul(s1, n, b1);
        mul(s2, a, b0);
        add(b1, s1, s2);
        
        //s1 = a1/b1;
        div(s1, a1, b1);
        if (!compare(s, s1))
        {
            //cout << "n = " << n << endl;
            break;
        }
        s = s1;
    }
    
    //res = s*exp(-a + x*log(a));
    log(s1, a);
    mul(s1, s1, x);
    sub(s1, s1, a);
    exp(s1, s1);
    mul(z, s, s1);
    
}/* largeIncompleteGamma */


// need to use optimum a for given argument given to small and large incomplete functions
// for x+n>1 use Gamma(x+n) = x(x+1)(x+2)...(x+n-1)Gamma(x) to reduce to x<1
// might use a = xt-1 wich maximizes t^(xt-1)*e(-t) as a function of t, so want 1<xt<2 ( Gamma(xt) = (xt-1)!so argument of factorial is positive )
// xMax avoids overflow in pow function - dropped
//86100000 gives 6.529519196446844e645811020
//86200000 gives 7.064028851869211e-646309441
void Gamma(fp& z, const fp& x)
{
    fp          a, zt, xt, one=1, xMax=86100000;
    fp          temp1, temp2;
    long        xInt, aInt, i, p, bp, r;
    
    if(compare(x, xMax)==1)
    {
        z = 0;
        return;
    }
    
    p = decPrec;
    bp = blockPrec;
    
    if(conv(xInt, x))
    {
        if(xInt<=0)
        {
            z = 0; // 0, -1, -2, ... not allowed
            return;
        }
        if(p<=64 && x<=20000)
        {
            zt = one;
            for(i=2;i<xInt;++i)
                zt = zt * i;
            fpNormalize(zt);
            z = zt;
            return;
        }
    }
    
    blockPrec++;
    decPrec+=16;
    
    if(x.i.s && compare(x, one)==1)
    {
        if(p<=64 && x>5000) // for speed
        {
            aInt = 70;
            GammaSpouge(zt, x-1, aInt);
            blockPrec = bp;
            decPrec = p;
            fpNormalize(zt);
            z = zt;
            return;
        }
        else
        {
            a = 200; // 100
            if(x>=100)
                a = 2*x;
            if(x>=1000)
                a = 3*x;
            if(x>=10000)
                a = 4*x;
            smallIncompleteGamma(temp1, x, a);
            largeIncompleteGamma(temp2, x, a);
            zt = temp1 + temp2;
            blockPrec = bp;
            decPrec = p;
            fpNormalize(zt);
            z = zt;
            return;
        }
    }
    // now x<1 and x is not a negative integer or zero so use reflection
    // for x+n>1 use Gamma(x+n) = x(x+1)(x+2)...(x+n-1)Gamma(x) to reduce to x<1
    // if x is <1 find integer n for which x+n is first >1
    // then Gamma(x) = Gamma(x+n)/[x(x+1)(x+2)...(x+n-1)]
    xInt = to_long(x);
    xInt = 2 - xInt; // xInt is one of 2, 3,
    xt = x + xInt; // xt>1
    a = 200;
    smallIncompleteGamma(temp1, xt, a);
    largeIncompleteGamma(temp2, xt, a);
    zt = temp1 + temp2;
    for(i=0;i<xInt;++i)
        zt = zt/(x+i);
    blockPrec = bp;
    decPrec = p;
    fpNormalize(zt);
    z = zt;
    
}/* Gamma */

fp Gamma(const fp& x)
{
    fp        z;
    
    Gamma(z, x);
    
    return z;
    
}/* Gamma */

void Gamma(fp& z, double x)
{
    fp        xt;
    
    xt = x;
    Gamma(z, xt);
    
}/* Gamma */


fp Gamma(double x)
{
    fp        z;
    
    Gamma(z, x);
    
    return z;
    
}/* Gamma */

