//
//  fpMath.cpp
//  fp64
//
//  Created by Bob Delaney on 12/10/18.
//  Copyright © 2018 Bob Delaney. All rights reserved.
//

#include <string.h>
#include "mbMath.hpp"
#include "fpMath.hpp"
#include "fpConv.hpp"
#include "fpFuncs.hpp"

long    decPrec=16;
long    blockPrec=1;
long    userPrec = 16;
long    outPrec = 16;
long    powCount=0;

static void coutHex(const mb& x)
{
    long    i;
    
    cout << "sign = " << x.s << endl;
    
    for(i=0;i<x.n;++i)
        cout << hex << x.b[i] << endl;
    cout << dec << endl;
    
}/* coutHex(mb) */

static void coutHex(const fp& x)
{
    cout << dec << "e = " << x.e << endl;
    coutHex(x.i);
    
}/* coutHex(fp) */

static void coutHex1(const mb& x)
{
    char        hexBlock[17];
    char        *hexString, ch, ch1;
    long        i, j, length;
    
    if(x.b==NULL)
        return;
    
    // x.n is the number of hex blocks to put into hexString
    // each block goes to 16 hex characters (or less)
    hexString = (char*)malloc((16*x.n+1)*sizeof(char));
    hexString[0] = 0;
    // use blocks from high to low
    // but must account for leading zeroes in the hex word!
    sprintf(hexBlock, "%lx", x.b[x.n-1]);
    strcat(hexString, hexBlock);
    for(i=x.n-2;i>=0;--i)
    {
        sprintf(hexBlock, "%lx", x.b[i]);
        length = strlen(hexBlock);
        for(j=0;j<16-length;++j)
            strcat(hexString, "0");
        
        strcat(hexString, hexBlock);
    }
    
    cout << "n = " << x.n << endl;
    cout << "sign = " << x.s << endl;
    cout << "hexString = " << hexString << endl;
    
}/* coutHex1 */



long getDecPrec()
{
    return decPrec;
    
}/* getDecPrec */

long getBlockPrec()
{
    return blockPrec;
    
}/* getBlockPrec */

void setUserPrec(long prec)
{
    long    r;
    
    if(prec<3)
        prec = 3;
    
    userPrec = prec;
    if(prec<16)
        prec = 16;
    blockPrec = prec/16;
    r = prec - 16*blockPrec;
    if(r>0)
        blockPrec++;
    decPrec = 16*blockPrec;
    
}/* setUserPrec */

long getUserPrec()
{
    return userPrec;
    
}/* getUserPrec */

void setOutPrec(long prec)
{
    if(prec<3)
        prec = 3;
    outPrec = prec;
    
}/* setOutPrec */

long getOutPrec()
{
    return outPrec;
    
}/* getOutPrec */

bool isItInteger(const fp& x)
{
    fp            xt;
    long        i;
    bool        isInteger;
    
    if(x.i==0)
        return true;
    
    xt = x;
    
    if(xt.e>=0)
        return true;
    
    // now xt.e<0 so might or might not have an integer
    
    if(xt.e<=-64)
        return false; // hex point is to left of block zero, which is not zero
    
    // now -63 <= xt.e <= -1
    // so using x.e block zero is of form ...xxx.xxxxx... say, where x is a binary bit
    // the binary point is abs(x.e) from the right
    // so we need to check the bits to the right of the binary point
    // if all are zero, we have an integer; if not, we don't
    
    isInteger = true;
    for(i=0;i<abs(xt.e);i++)
    {
        if(xt.i.b[0] & 1)
        {
            isInteger = false;
            break;
        }
        xt.i.b[0] = (xt.i.b[0]>>1);
    }
    
    return isInteger;
    
}/* isItInteger */

bool isItEven(const fp& x)
{
    mb  xt;
    
    if(x.e>0)
        return true;
    
    xt = x.i;
    
    if(xt.b[0] & 1)
        return false;
    
    return true;
    
}/* isItEven */

// z = floor(x)
void floor(fp& z, const fp& x)
{
    fp              xt;
    fp              one=1;
    
    xt = x;;
    
    if(xt.e>=0) // it already is an integer
    {
        z = xt;;
        return;
    }
    
    // now xt.e<0
    mbShiftRight(xt.i, xt.i, abs(xt.e));
    xt.e = 0;
    
    if(!xt.i.s)
        sub(xt, xt, one);
    
    if(xt.i.b[0]==0 && !x.i.s)
        sub(xt, xt, one);
    
    z = xt;
    
}/* floor */

fp floor(const fp& x)
{
    fp        z;
    
    floor(z, x);
    return z;
    
}/* floor */

// z = ceil(x)
void ceil(fp& z, const fp& x)
{
    fp            xt;
    
    xt = x;
    
    if(xt.e>=0) // it already is an integer
    {
        z = xt;
        return;
    }
    
    // now xt.e<0
    mbShiftRight(xt.i, xt.i, abs(xt.e));
    xt.e = 0;
    
    if(xt>0)
        xt = xt + 1;
    
    if(xt==0 && x.i.s)
        xt = xt + 1;
    
    z = xt;
    
}/* ceil */

fp ceil(const fp& x)
{
    fp        z;
    
    ceil(z, x);
    return z;
    
}/* ceil */

// converts x to a long; it assumes x is in proper range
long to_long(const fp& x)
{
    fp                  sx;
    long                z;
    
    removeTrailingZeroes(sx, x);
    
    if(sx.e>=0)
    {
        mbShiftLeft(sx.i, sx.i, sx.e);
    }
    else
    {
        mbShiftRight(sx.i, sx.i, abs(sx.e));
    }
    sx.e = 0;
    /*
    if(!sx.i.n)
        return 0;
     */
    
    z = sx.i.b[0];
    if(!x.i.s)
        z = -z;
    
    return z;
    
}/* to_long */

// remove low order zero blocks, then zero block low order zeroes
void removeTrailingZeroes(fp& z, const fp& x)
{
    fp          zt;
    long        i, bitsShift, numZeroblocks;
    
    zt = x;
    numZeroblocks = 0;
    for(i=0;i<zt.i.n;++i)
    {
        if(zt.i.b[i]==0)
            numZeroblocks++;
        else
            break;
    }
    
    if(numZeroblocks)
        fpShiftRightmb(zt, zt, numZeroblocks*blockBits);
    
    bitsShift = numLowOrderZeroBits(zt.i.b[0]);
    mbShiftRight(zt.i, zt.i, bitsShift);
    zt.e+=bitsShift;
    mbNormalize(zt.i);
    z = zt;
    
}/* removeTrailingZeroes */

fp abs(const fp& x)
{
    fp        xt;
    
    xt = x;
    if(xt.i.s==false)
        xt.i.s = true;
    return xt;
    
}/* abs */

// returns approximate log(abs(x)) to base 2
long Lg2(const fp& x)
{
    fp                  xt;
    static UINT64       mask;
    static bool         initGood=false;
    long                i, xPowerOfTwo, numZeroBits;
    long                cs;
    
    if(!initGood)
    {
        mask = 1;
        mask = (mask<<(blockBits-1)); // 0x8000000000000000
        initGood = true;
    }
    
    if(!x.i.n)
        return 0; // bad input
    
    xt = x;
    xPowerOfTwo = xt.e + blockBits*(xt.i.n-1);
    
    // look at high blocks
    numZeroBits = 0;
    cs = xt.i.b[xt.i.n-1];
    for(i=0;i<blockBits;i++)
    {
        if(!(cs & mask))
        {
            numZeroBits++;
            cs = (cs<<1);
        }
        else
            break;
    }
    
    return xPowerOfTwo + blockBits - numZeroBits - 1;
    
}/* Lg2 */

// multiplies by 2^numBits
void fpShiftLeft(fp& z, const fp& x, long numBits)
{
    fp      zt;
    
    zt = x;
    mbShiftLeft(zt.i, zt.i, numBits);
    zt.e = x.e;
    z = zt;
    
}/* fpShiftLeft */

// divides by 2^numBits
void fpShiftRight(fp& z, const fp& x, long numBits)
{
    fp      zt;
    
    zt = x;
    mbShiftRight(zt.i, zt.i, numBits);
    zt.e = x.e;
    z = zt;
    
}/* fpShiftRight */

// shifts left x.i.b and changes e so have same number
void fpShiftLeftmb(fp& z, const fp& x, long numBits)
{
    fp      zt;
    
    zt = x;
    mbShiftLeft(zt.i, zt.i, numBits);
    zt.e = x.e;
    zt.e-=numBits;
    z = zt;
    
}/* fpShiftLeftmb */

// shifts right x.i.b and changes e so have same number
void fpShiftRightmb(fp& z, const fp& x, long numBits)
{
    fp      zt;
    
    zt = x;
    mbShiftRight(zt.i, zt.i, numBits);
    zt.e = x.e;
    zt.e+=numBits;
    z = zt;
    
}/* fpShiftRightmb */

// compares abs(x) to abs(y); returns:
// +1 if abs(x)>abs(y)
// 0  if abs(x)=abs(y)
// -1 if abs(x)<abs(y)
static long fpCompareAbs(const fp& x, const fp& y)
{
    fp              xt, yt;
    long            xPowerOfTwo, yPowerOfTwo;
    long            i, bitShift;
    
    if(x.i==0 && y.i==0)
        return 0;
    
    if(x.i==0)
        return -1;
    
    if(y.i==0)
        return 1;
    
    xt = x;
    yt = y;
    
    // coarse compare first
    xPowerOfTwo = xt.e + blockBits*(xt.i.n-1) + NumBits(xt.i.b[xt.i.n-1]);
    yPowerOfTwo = yt.e + blockBits*(yt.i.n-1) + NumBits(yt.i.b[yt.i.n-1]);
    
    if(xPowerOfTwo>yPowerOfTwo)
        return 1;
    
    if(xPowerOfTwo<yPowerOfTwo)
        return -1;
    
    // now xPowerOfTwo = yPowerOfTwo
    // we need the most sig bit of both xt and yt to be highest order bit of highest order block
    // so we can compare block by block from highest to lowest
    
    bitShift = blockBits - NumBits(xt.i.b[xt.i.n-1]);
    mbShiftLeft(xt.i, xt.i, bitShift);
    xt.e-=bitShift; // not needed
    
    bitShift = blockBits - NumBits(yt.i.b[yt.i.n-1]);
    mbShiftLeft(yt.i, yt.i, bitShift);
    yt.e-=bitShift; // not needed
    
    if(xt.i.n>=yt.i.n)
    {
        for(i=xt.i.n-1;i>=(xt.i.n-yt.i.n);i--)
        {
            if(xt.i.b[i]>yt.i.b[yt.i.n-xt.i.n+i])
                return 1;
            else
                if(xt.i.b[i]<yt.i.b[yt.i.n-xt.i.n+i])
                    return -1;
        }
        
        for(i=xt.i.n-yt.i.n-1;i>=0;i--)
        {
            if(xt.i.b[i]>0)
                return 1;
        }
        
        return 0;
    }
    else
    {
        for(i=yt.i.n-1;i>=yt.i.n-xt.i.n;i--)
        {
            if(yt.i.b[i]>xt.i.b[xt.i.n-yt.i.n+i])
                return -1;
            else
                if(yt.i.b[i]<xt.i.b[xt.i.n-yt.i.n+i])
                    return 1;
        }
        
        for(i=yt.i.n-xt.i.n-1;i>=0;i--)
        {
            if(yt.i.b[i]>0)
                return -1;
        }
        
        return 0;
    }
    
    return 0;
    
}/* compareAbs(fp, fp) */

/*
 assumes x an y are normalized
 returns:
 1 if x>y
 0 if x=y
 -1 if x<y
 */
long compare(const fp& x, const fp& y)
{
    long    result;

    result = fpCompareAbs(x, y);
    
    if(!result)
    {
        // worry about comparing 0 and -0
        if(x.i.n==1 && y.i.n==1 && x.i.b[0]==0)
            return 0;
        
        if(x.i.s && !y.i.s)
            return 1;
        
        if(!x.i.s && y.i.s)
            return -1;
        
        return 0;
    }
    
    if(result==1)
    {
        if(x.i.s)
            return 1;
        else
            return -1;
    }
    
    // now result = -1
    if(y.i.s)
        return -1;
    else
        return 1;
    
}/* compare */

// this removes leading and trailing zeroblocks from the integer part and then if has more blocks than blockPrec, it trims down
// we need to make sure that the highest order block is as large as possible by using mbShiftLeft
void fpNormalize(fp& x)
{
    static UINT128          mask1; // 0x10000000000000000
    static UINT128          mask2; // 0xFFFFFFFFFFFFFFFF
    static UINT128          cxMin;
    static bool             initGood=false;
    long                    numBlocks, numZeroBlocks, numZeroBits, bitsShift;
    bool                    isAllFs;
    long                    i;
    UINT128                 cx, carry;
    fp                      xt;
    
    if(!initGood)
    {
        mask1 = 1;
        mask1 = (mask1<<blockBits);
        mask2 = mask1 - 1;
        cxMin = 0x7FFFFFFFFFFFFFFF;
        initGood = true;
    }
    
    if(x.i==0)
    {
        x.e = 0;
        return;
    }
    
    xt = x;
    
    if(xt.i.n<=blockPrec) // there may be trailing zeroes
    {
        numZeroBlocks = 0;
        for(i=0;i<xt.i.n;i++)
        {
            if(!xt.i.b[i])
                numZeroBlocks++;
            else
                break;
        }
        
        if(numZeroBlocks)
        {
            bitsShift = blockBits*numZeroBlocks;
            fpShiftRightmb(xt, xt, bitsShift);
        }
        x = xt;
        return;
    }
    
    // shift left to get largest high order block
    numZeroBits = alignLeft(xt.i);
    xt.e-=numZeroBits;
    
    // remove trailing zero blocks
    numZeroBlocks = 0;
    for(i=0;i<xt.i.n;i++)
    {
        if(!xt.i.b[i])
            numZeroBlocks++;
        else
            break;
    }
    
    if(numZeroBlocks)
    {
        bitsShift = blockBits*numZeroBlocks;
        fpShiftRightmb(xt, xt, bitsShift);
    }
    
    if(xt.i==0)
    {
        xt.e = 0;
        x = xt;
        return;
    }
    
    if(xt.i.n<=blockPrec)
    {
        x = xt;
        return;
    }
    
    // now x.i.n>blockPrec so we need to round up and trim
    // but have to worry about the leading blocks some or all
    // being FFFFFFFFFFFFFFFF.
    // the lowest block just below the blockPrec ones has index x.i.n-blockPrec-1
    // if its value is >0x7FFFFFFFFFFFFFFF we need to round, else just trim
    cx = xt.i.b[xt.i.n-blockPrec-1];
    if(cx>cxMin)
    {
        // special case of all blockPrec values being FFFFFFFFFFFFFFFF
        isAllFs = true;
        for(i=xt.i.n-1;i>=xt.i.n-blockPrec;i--)
            if(xt.i.b[i]!=mask2)
            {
                isAllFs = false;
                break;
            }
        if(isAllFs)
        {
            xt.e+=(blockBits*(xt.i.n));
            xt.i.n = 1;
            xt.i.b[0] = 1;
            x = xt;
            return;
        }
        // now must add 1 to blockPrec part
        carry = 1;
        for(i=xt.i.n-blockPrec;i<xt.i.n;i++)
        {
            cx = xt.i.b[i];
            cx = cx + carry;
            carry = ((cx & mask1)>>blockBits);
            xt.i.b[i] = (cx & mask2);
            if(!carry)
                break;
        }
        // trim trailing zeroes
        numZeroBlocks = 0;
        for(i=0;i<xt.i.n;i++)
        {
            if(!xt.i.b[i])
                numZeroBlocks++;
            else
                break;
        }
        
        if(numZeroBlocks)
        {
            bitsShift = blockBits*numZeroBlocks;
            fpShiftRightmb(xt, xt, bitsShift);
        }
        
        if(xt.i==0)
        {
            xt.e = 0;
            x = xt;
            return;
        }
    }
    
    //blockPrec++;
    // now trim if necessary
    numBlocks = xt.i.n - blockPrec;
    if(numBlocks>0)
    {
        bitsShift = blockBits*numBlocks;
        fpShiftRightmb(xt, xt, bitsShift);
    }
    //blockPrec--;
    x = xt;
    
}/* fpNormalize */

// z = x + y
void add(fp& z, const fp& x, const fp& y)
{
    fp              xt, yt, zt;
    long            eDiff;
    long            s;
    long            lg2x, lg2y;
    long            eSave;
    
    if(x.i==0)
    {
        z = y;
        return;
    }
    
    if(y.i==0)
    {
        z = x;
        return;
    }
    
    xt = x;
    yt = y;
    
    lg2x = Lg2(xt);
    lg2y = Lg2(yt);
    
    if(abs(lg2x-lg2y)>((blockPrec+1)*blockBits))
    {
        if(lg2x>lg2y)
        {
            fpNormalize(xt);
            zt = xt;
            z = zt;
            return;
        }
        else
        {
            fpNormalize(yt);
            zt = yt;
            z = zt;
            return;
        }
    }
    
    if(xt.e==yt.e)
    {
        eSave = xt.e;
        add(zt.i, xt.i, yt.i);
        zt.e = eSave;
        fpNormalize(zt);
        z = zt;
        return;
    }
    
    if(xt.e>yt.e)
    {
        eDiff = xt.e - yt.e;
        fpShiftLeftmb(xt, xt, eDiff);
        eSave = xt.e;
        add(zt.i, xt.i, yt.i);
        zt.e = eSave;
        fpNormalize(zt);
        z = zt;
        return;
    }
    // now xt.e < yt.e
    eDiff = yt.e - xt.e;
    fpShiftLeftmb(yt, yt, eDiff);
    eSave = yt.e;
    add(zt.i, xt.i, yt.i);
    zt.e = eSave;
    fpNormalize(zt);
    z = zt;

}/* add(fp, fp, fp) */

// z = x - y
void sub(fp& z, const fp& x, const fp& y)
{
    fp              xt, yt, zt;
    long            eDiff;
    long            s;
    long            lg2x, lg2y;
    long            eSave;
    
    if(x.i==0)
    {
        z = -y;
        return;
    }
    
    if(y.i==0)
    {
        zt = x;
        z = zt;
        return;
    }
    
    xt = x;
    yt = y;
    
    lg2x = Lg2(xt);
    lg2y = Lg2(yt);
    
    if(abs(lg2x-lg2y)>((blockPrec+1)*blockBits))
    {
        if(lg2x>lg2y)
        {
            zt = xt;
            z = zt;
            fpNormalize(z);
            return;
        }
        else
        {
            z = -yt;
            fpNormalize(z);
            return;
        }
    }
    
    if(xt.e==yt.e)
    {
        eSave = xt.e;
        sub(zt.i, xt.i, yt.i);
        zt.e = eSave;  // xt loses xt.e, so did zt
        fpNormalize(zt);
        z = zt;
        return;
    }
    
    if(xt.e>yt.e)
    {
        eDiff = xt.e - yt.e;
        fpShiftLeftmb(xt, xt, eDiff);
        eSave = xt.e;
        sub(zt.i, xt.i, yt.i);
        zt.e = eSave;
        fpNormalize(zt);
        z = zt;
        return;
    }
    // now xt.e < yt.e
    eDiff = yt.e - xt.e;
    fpShiftLeftmb(yt, yt, eDiff);
    eSave = yt.e;
    sub(zt.i, xt.i, yt.i);
    zt.e = eSave;
    fpNormalize(zt);
    z = zt;
}/* sub */

// z = x*y
void mul(fp& z, const fp& x, const fp& y)
{
    fp      zt;
    
    if(x.i==0 || y.i==0)
    {
        z = 0;
        return;
    }
    
    mul(zt.i, x.i, y.i);
    zt.e = x.e + y.e;
    fpNormalize(zt);
    z = zt;
    
}/* mul */

// z = x/y
// we have to worry about going to blockPrec for unending divisions
bool div(fp& z, const fp& x, const fp& y)
{
    fp              s, yt, zt;
    long            blocksShift, bitsShift;
    long            eSave;
    
    if(y.i==0)
        return false;
    
    if(x.i==0)
    {
        z = 0;
        return true;
    }
    
    s = x;
   
    // shift s.i left so new s.i.n = blockPrec + y.i.n + 1
    // may have to increase s.i.n, but mbShiftLeft takes care of that
    blocksShift = blockPrec - x.i.n + y.i.n + 1;
    if(blocksShift>0)
    {
        bitsShift = blockBits*blocksShift;
        fpShiftLeftmb(s, s, bitsShift);
    }
    yt = y;
    eSave = s.e;
    div(zt.i, s.i, yt.i);
    s.e = eSave;
    zt.e = s.e - y.e;
    fpNormalize(zt);
    bitsShift = numLowOrderZeroBits(zt.i.b[0]);
    fpShiftRightmb(zt, zt, bitsShift);
    mbNormalize(zt.i);
    z = zt;
    return true;
    
}/* div */

void add(fp& z, const fp& x, double y)
{
    fp      yfp;
    
    conv(yfp, y);
    add(z, x, yfp);
    
}/* add(fp, fp, double) */

void sub(fp& z, const fp& x, double y)
{
    fp      yfp;
    
    conv(yfp, y);
    sub(z, x, yfp);
    
}/* sub(fp, fp, double) */

void mul(fp& z, const fp& x, double y)
{
    fp      yfp;
    
    conv(yfp, y);
    mul(z, x, yfp);
    
}/* mul(fp, fp, double) */

void div(fp& z, const fp& x, double y)
{
    fp      yfp;
    
    conv(yfp, y);
    div(z, x, yfp);
    
}/* div(fp, fp, double) */

void add(fp& z, double x, const fp& y)
{
    fp      xfp;
    
    conv(xfp, x);
    add(z, xfp, y);
    
}/* add(fp, double, fp) */

void sub(fp& z, double x, const fp& y)
{
    fp      xfp;
    
    conv(xfp, x);
    sub(z, xfp, y);
    
}/* sub(fp, double, fp) */

void mul(fp& z, double x, const fp& y)
{
    fp      xfp;
    
    conv(xfp, x);
    mul(z, xfp, y);
    
}/* mul(fp, double, fp) */

void div(fp& z, double x, const fp& y)
{
    fp      xfp;
    
    conv(xfp, x);
    div(z, xfp, y);
    
}/* div(fp, double, fp) */

// z = x^y ; y is a long
/*
 Uses Victor Shoup's NTL method.
 If bit(i) is ith bit of y, i = 0, 1, ... , 30 (bit(31) is sign bit)
 then abs(y) = Sum(i=0 to 30) bit(i)*2^i
 so x^y = x^(bit(0)) * x^(bit(1)*2) * x^(bit(2)*4) * x^(bit(3)*8) * x^(bit(4)*16) * ...
 Note that if all bit(i)=1 each term in the product is the square of the previous term
 So we remember that and multiply in if bit(i)=1, but don't multiply in if bit(i)=0
 Then Shoup does that in an ingenious way
 NumBits(y) gives the number of bits after and including the high order 1 bit, not counting the sign bit
 */
bool power(fp& z, const fp& x, long y)
{
    static fp               one;
    fp                      zt;
    static UINT64           mask; // 0x8000000000000000
    static bool             initGood=false;
    long                    i, n, sy;
    
    if(!initGood)
    {
        one = 1;
        mask = 1;
        mask = (mask<<(blockBits-1)); // 0x8000000000000000
        initGood = true;
    }
    
    if(x.i==0)
    {
        z = 0;
        if(y<=0)
            return false;
        
        return true;
    }
    
    if(y==0)
    {
        z = one;
        return true;
    }
    
    zt = one;
    sy = y;
    if(y<0)
        sy = -y;
    n = NumBits(sy);
    
    sy = (sy<<(64-n));  // go to leading 1 bit
    
    for(i=0;i<n;i++)
    {
        mul(zt, zt, zt); // zt^2
        if(sy & mask)
            mul(zt, zt, x);
        sy = (sy<<1);
    }
    
    if(y<0)
        div(zt, one, zt);
    
    z = zt;
    
    return true;
    
}/* power */

fp power(const fp& x, long y)
{
    fp  z;
    
    if(!power(z, x, y))
        z = 0;
    return z;
    
}/* power */

// z = x^y
bool pow(fp& z, const fp& x, const fp& y)
{
    static fp           one=1;
    fp                  xt, yt, zt;
    long                yInt;
    
    if(y.i==0)
    {
        z = one; // so 0^0 = 1
        return true;
    }
    
    if(x.i==0)
    {
        z = 0; // the caller has to check for y=0 for this case
        return true;
    }
    
    if(!compare(x, one))
    {
        zt = one;
        z = zt;
        return true;
    }
    
    if(conv(yInt, y))
    {
        // y is an integer
        power(zt, x, yInt);
        z = zt;
        return true;
    }
    
    xt = x;
    
    if(!isItInteger(y) && !x.i.s)
    {
        zt = 0;
        z = zt;
        return false;
    }
    
    if(y.i==0)
    {
        zt = one;
        z = zt;
        return true;
    }
    
    if(isItInteger(y))
    {
        power(zt, x, to_long(y));
        z = zt;
        return true;
    }
    
    log(zt, xt);
    mul(zt, zt, y);
    exp(zt, zt);
    z = zt;
    
    return true;
    
}/* pow */

fp pow(const fp& x, const fp& y)
{
    fp        z;
    
    pow(z, x, y);
    return z;
}/* pow */

void pow(fp& z, const fp& x, double  y)
{
    fp  yfp;
    
    yfp = y;
    pow(z, x, yfp);
    
}/* pow(fp, fp, double) */

fp pow(const fp&x, double y)
{
    fp  z;
    
    pow(z, x, y);
    return z;
    
}/* pow(fp, double) */
