//
//  fpFuncs.cpp
//  fp64
//
//  Created by Bob Delaney on 12/13/18.
//  Copyright © 2018 Bob Delaney. All rights reserved.
//

#include "fp.hpp"
#include "fpMath.hpp"
#include "fpConv.hpp"
#include "fpFuncs.hpp"
#include "mbMath.hpp"
#include "myGammaFunctions.hpp"

extern long blockPrec, decPrec;
extern long powCount;

static void coutHex(const mb& x)
{
    long    i;
    
    cout << "sign = " << x.s << endl;
    
    for(i=0;i<x.n;++i)
        cout << hex << x.b[i] << endl;
    cout << dec << endl;
    
}/* coutHex(mb) */

static void coutHex(const fp& x)
{
    cout << "e = " << x.e << endl;
    coutHex(x.i);
    
}/* coutHex(fp) */

// a fast z = 2^n
static void power2(fp& z, long n)
{
    static fp           one;
    static bool         initGood=false;
    
    if(!initGood)
    {
        init(one, 1);
        one.i.b[0] = 1;
        one.e = 0;
        initGood = true;
    }
    
    one.e = n; // one is now 2^n
    z = one;
    
}/* power2 */

static fp power2(long n)
{
    fp        z;
    
    power2(z, n);
    return z;
    
}/* power2 */

// z = ln(2)
static void ComputeLn2(fp& z)
{
    static fp       pointFive, Ln2;
    fp              s, s1, t, t1;
    static long     blockPrecCurr;
    static bool     initGood=false;
    long            i;
    fp              ifp;
    
    if(!initGood)
    {
        blockPrecCurr = 0;
        init(pointFive, 1);
        pointFive.i.b[0] = 1;
        pointFive.e = -1;
        initGood = true;
    }
    
    if(blockPrec<=blockPrecCurr)
    {
        z = Ln2;
        fpNormalize(z);
        return;
    }
    
    blockPrecCurr = blockPrec;
    
    t = pointFive;
    t1 = pointFive;
    s = 0;
    
    for (i=2; ; i++)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t1, t1, pointFive);
        ifp = i;
        div(t, t1, ifp);
    }
    z = s;
    Ln2 = s;
    
}/* ComputeLn2 */

// log emulates Victor Shoup's NTL method
void log(fp& z, const fp& x)
{
    fp                  ifp;
    static fp           pointFive, pointSevenFive, onePointFive, one, two, y, s, s1, t, t1;
    static bool         initGood=false;
    long                i, n, result;
    
    if(!initGood)
    {
        init(one, 1);
        one.i.b[0] = 1;
        one.e = 0;
        init(two, 1);
        two.i.b[0] = 2;
        two.e = 0;
        
        init(pointFive, 1);
        pointFive.i.b[0] = 1;
        pointFive.e = -1;
        init(onePointFive, 1);
        onePointFive.i.b[0] = 3;
        onePointFive.e = -1;
        init(pointSevenFive, 1);
        pointSevenFive.i.b[0] = 3;
        pointSevenFive.e = -2;
        // the above are all exact, so blockPrec doesn't apply
        initGood = true;
    }
    
    if(!x.i.s)
        return;
    
    // re-write x = 2^n * (1 - y), where -1/2 < y < 1/4  (so 3/4 < 1-y < 3/2)
    result = compare(x, pointSevenFive);
    
    if((result==1 || result==0) && compare(x, onePointFive)==-1)
    {
        n = 0;
        sub(y, one, x);
    }
    else
    {
        n = Lg2(x);
        power2(t, -n);
        mul(t, t, x);
        
        while(compare(t, onePointFive)==1)
        {
            mul(t, t, pointFive);
            n++;
        }
        sub(y, one, t);
    }
    
    // compute s = - ln(1-y) by power series expansion
    s =  0;
    //s1 = 0;
    t = y;
    t1 = y;
    
    for(i=2; ;i++)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t1, t1, y);
        ifp = i;
        div(t, t1, ifp);
    }
    
    if(n==0)
    {
        t = 0;
    }
    else
    {
        ComputeLn2(t);
        mul(t, t, abs(n));
        if(n<0)
            t.i.s = !t.i.s;
    }
    
    sub(z, t, s);
    
}/* log */

fp log(const fp& x)
{
    fp  z;
    
    log(z, x);
    return z;
    
}/* log */


// z = sqrt(x)
void sqrt(fp& z, const fp& x)
{
    fp      zt, ztS, two=2;
    long    i;
    
    zt = x;
    ztS = 1;
    for(i=0; ;i++)
    {
        zt = (zt + x/zt)/two;
        if(zt==ztS)
            break;
        ztS = zt;
    }
    z = zt;
    
}/* sqrt */

fp sqrt(const fp& x)
{
    fp  z;
    
    sqrt(z, x);
    return z;
    
}/* sqrt */

void sqr(fp& z, const fp& x)
{
    
    mul(z, x, x);
    
}/* sqr */

fp sqr(const fp& x)
{
    fp  z;
    
    sqr(z, x);
    return z;
    
}/* sqr */

void ComputeE(fp& z)
{
    static fp               one, e;
    fp                      s, s1, t;
    static long             blockPrecCurr;
    static bool             initGood=false;
    long                    i;
    
    if(!initGood)
    {
        one = 1;
        blockPrecCurr = 0;
        initGood = true;
    }
    
    if(blockPrec<=blockPrecCurr)
    {
        z = e;
        return;
    }
    
    blockPrecCurr = blockPrec;
   
    s = one;
    t = one;
    for(i=2;;i++)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        div(t, t, i);
    }
    
    e = s;
    z = s;
    
}/* ComputeE */

// exp emulates Victor Shoup's NTL method
void exp(fp& z, const fp& x)
{
    fp                  one=1;
    fp                  f, nn, e, t, t1, t2, s, s1;
    long                i, n;
    
    if(abs(Lg2(x))>.69*0x7FFFFFFFFFFFFFFF)
        return; //avoids overflow
    
    floor(nn, x);
    
    sub(f, x, nn);
    
    // convert nn to INT32
    n = to_long(nn);
    
    // step 1: calculate t1 = e^n by repeated squaring
    
    ComputeE(e);
    
    power(t1, e, n); // offender n = 6950
    
    // step 2: calculate t2 = e^f using Taylor series expansion
    
    s = 0;
    t = one;
    
    for(i=1; ;i++)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t, t, f);
        div(t, t, i);
    }
    
    t2 = s;
   
    mul(s, t1, t2);
    
    z = s;
    
}/* exp */

fp exp(const fp& x)
{
    fp  z;
    
    exp(z, x);
    return z;
    
}/* exp */

// uses Bailey-Borwein-Plouffe algorithm
void ComputePi(fp& z)
{
    static fp               one=1, two=2, four=4, s47=47, s15=15, oneSixteenth, pi;
    fp                      s, s1, t, t1, t2;
    static long             blockPrecCurr;
    static bool             initGood=false;
    long                    i;
    
    if(!initGood)  // the calculation of pi is much slower if we don't init like this
    {
        init(oneSixteenth, 1);
        oneSixteenth.e = -4;   // if use oneSixteenth(.0625) get a 5X slowdown!
        oneSixteenth.i.b[0] = 1;
        oneSixteenth.i.s = true;
        blockPrecCurr = 0;
        initGood = true;
    }
    
    if(blockPrec<=blockPrecCurr)
    {
        z = pi;
        fpNormalize(z); // assume decPrec has already been changed
        return;
    }
    
    blockPrecCurr = blockPrec;
    
    s = s47;
    div(s, s, s15); // i = 0 term
    s1 = s;
    
    t = one; // powerFactor
   
    for(i=1; ;i++)
    {
        mul(t, t, oneSixteenth);
        t1 = four;
        div(t1, t1, 8*i+1);
        t2 = two;
        div(t2, t2, 8*i+4);
        sub(t1, t1, t2);
        div(t2, one, 8*i+5);
        sub(t1, t1, t2);
        div(t2, one, 8*i+6);
        sub(t1,t1,t2); // problem here
        mul(t1, t1, t);
        add(s, s, t1);
        if(!compare(s, s1))
            break;
        s1 = s;
    }
    
    pi = s;
    z = s;
    
}/* ComputePi */

fp Pi()
{
    fp  z;
    
    ComputePi(z);
    return z;
    
}/* Pi */

// e.c. = -Gamma'(1)
static void euler(fp& res)
{
    long        userPrecision, bitPrecision;
    fp          dx, dx2, ten, one, temp, tenPower, small;
    
    long p = blockPrec;
    blockPrec = p + p/4;
    
    one = 1;
    ten = 10;
    
    bitPrecision = blockBits*p;
    userPrecision = bitPrecision/5;
    tenPower = 2.09*userPrecision/5;
    pow(dx, ten, -tenPower);
    
    Gamma(temp, one + dx);
    small = temp;
    
    Gamma(temp, one - dx);
    small = small - temp;
    
    dx2 = 2*dx;
    
    Gamma(temp, one + dx2);
    res = - temp/8;
    
    Gamma(temp, one - dx2);
    res = res + temp/8;
    
    res = small + res;
    
    res = -2*res/(3*dx);
    
    blockPrec = p;
    
}/* euler */

// Euler's Constant
void EulerGamma(fp& z)
{
    static fp           gamma;
    static bool         initGood=false;
    static long         p, blockPrecCurr;
    
    if(!initGood)
    {
        blockPrecCurr = 0;
        initGood = true;
    }
    
    p = getBlockPrec();
    
    if(p<=blockPrecCurr)
    {
        z = gamma;
        fpNormalize(z);
        return;
    }
    
    blockPrecCurr = p;
    
    euler(z);
    
    gamma = z;
    
}/* EulerGamma */


fp EulerGamma()
{
    fp        z;
    
    EulerGamma(z);
    
    return z;
    
}/* EulerGamma */

// sin emulates Victor Shoup's NTL method with big changes
void sin(fp& z, const fp& x)
{
    static fp       one=1, three=3, pointFive;
    fp              f, n, t, t1, t2, s, s1, x2;
    static bool     initGood=false;
    long            i;
    
    if(!initGood)
    {
        init(pointFive, 1);
        pointFive.e = -1;
        pointFive.i.s = true;
        pointFive.i.b[0] = 1;
        initGood = true;
    }
    
    if(x.i==0)
    {
        z = 0;
        return;
    }
    
    mul(x2, x, x); // x*x
    
    if(compare(x2, three)==-1)
    {
        f = x;
    }
    else
    {
        div(t1, x, Pi());
        floor(n, t1);
        sub(f, t1, n);
        if(compare(f, pointFive)==1)
        {
            add(n, n, one);
            sub(f, t1, n);
        }
        
        mul(f, Pi(), f);
        
        if(n.i.n && !isItEven(n))
        {
            if(f.i.s)
                f.i.s = false;
            else
                f.i.s = true;
        }
    }
    
    t = f;
    s = 0;
    fp        fSqr;
    fSqr = sqr(f);
     
    for(i=3; ;i+=2)
    {
        add(s1, s, t);
        /*
        if(powCount==30002)
        {
            coutHex(s);
            coutHex(s1);
        }
         */
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t, t, fSqr);
        div(t, t, i*(i-1));
        t.i.s = !t.i.s;
    }
    
    z = s;
    
}/* sin */


fp sin(const fp& x)
{
    fp        z;
    
    sin(z, x);
    return z;
    
}/* sin */

// cos emulates Victor Shoup's NTL method
void cos(fp& z, const fp& x)
{
    static fp           pointFive, one=1, three=3;
    fp                  f, n, t, t1, s, s1, temp;
    static bool         initGood=false;
    long                i;
    
    if(!initGood)
    {
        init(pointFive, 1);
        pointFive.e = -1;
        pointFive.i.s = true;
        pointFive.i.b[0] = 1;
        initGood = true;
    }
    
    if(x.i==0)
    {
        z = 1;
        return;
    }
    
    div(t1, x, Pi());
    floor(n, t1);
    add(temp, n, pointFive);
    sub(f, t1, temp);
    mul(f, Pi(), f);
    
    if(n.i==0 || isItEven(n))
    {
        if(f.i.s)
            f.i.s = false;
        else
            f.i.s = true;
    }
    
    t = f;
    s = 0;
    
    fp        fSqr;
    fSqr = sqr(f);
    
    for(i=3; ;i+=2)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t, t, fSqr);
        div(t, t, i*(i-1));
        if(t.i.s)
            t.i.s = false;
        else
            t.i.s = true;
    }
    
    z = s;
    
    return;
    
}/* cos */


fp cos(const fp& x)
{
    fp        z;
    
    cos(z, x);
    return z;
    
}/* cos */

void tan(fp& z, const fp& x)
{
    fp      myS, myC;
    
    cos(myC, x);
    if(myC.i==0)
        return;
    
    sin(myS, x);
    div(z, myS, myC);
}/* tan */

fp tan(const fp& x)
{
    fp        z;
    
    tan(z, x);
    return z;
    
}/* tan */

void atan(fp& z, const fp& x)
{
    fp                  one=1, four=4, pointFive, pointZeroOne;
    fp                  theTan, yPrev, y, y2, currentTerm, pi, temp;
    static bool         initGood=false;
    long                numReductions;
    bool                isReciprocal, isNegative;
    long                i;
    
    if(!initGood)
    {
        init(pointFive, 1);
        pointFive.e = -1;
        pointFive.i.s = true;
        pointFive.i.b[0] = 1;
        initGood = true;
    }
    pointZeroOne = ".01";
    
    numReductions = 0;
    
    if(x.i==0)
    {
        z = 0;
        return;
    }
    
    theTan = x;
    if(!theTan.i.s)
    {
        theTan.i.s = true;
        isNegative = true;
    }
    else
        isNegative = false;
    
    if(!compare(theTan, one))
    {
        ComputePi(pi);
        div(z, pi, four);
        if(isNegative)
            z.i.s = false;

        return;
    }
    
    if(compare(theTan, one)==1)
    {
        div(theTan, one, theTan);
        isReciprocal = true;
    }
    else
        isReciprocal = false;
    
    // divides arctan by a factor of two for each iteration
    while(compare(theTan, pointZeroOne)==1)
    {
        div(y, one, theTan);
        mul(y2, y, y);
        add(y2, y2, one);
        sqrt(y2, y2);
        add(y2, y2, y);
        div(theTan, one, y2);
        numReductions++;
    }
    
    // series method
    // atan z = z - z^3/3 + z^5/5 - z^7/7 + ...
    y = theTan;
    yPrev = y;
    currentTerm = y;
    mul(y2, y, y);
    y2.i.s = false;
    
    for(i=0;;i+=2)
    {
        mul(currentTerm, currentTerm, y2);
        div(temp, currentTerm, i+3);
        add(y, y, temp);
        if(!compare(y, yPrev))
            break;
        yPrev = y;
    }
    
    temp = one;
    temp.e+=numReductions;
    mul(y, y, temp);
    
    if(isReciprocal)
    {
        ComputePi(pi);
        pi.e--; // pi/2
        sub(y, pi, y);
    }
    
    if(isNegative)
    {
        if(y.i.s)
            y.i.s = false;
        else
            y.i.s = true;
    }
    
    z = y;
    
}/* atan */


fp atan(const fp& x)
{
    fp      z;
    
    atan(z, x);
    return z;
    
}/* atan */

// z = atan(y/x) in correct quadrant
void atan2(fp& z, const fp& y, const fp& x)
{
    fp          myPi, myTan;
    fp          two="2";
    
    if(x.i==0 && y.i==0)
    {
        z = 0; // actually undetermined
        return;
    }
    
    ComputePi(myPi);
    
    if(x.i==0)
    {
        z = myPi;
        div(z, z, two);
        if(!y.i.s)
            z.i.s = false;
        return;
    }
    
    if(y.i==0)
    {
        if(x.i.s)
            z = 0;
        else
            z = myPi;
        return;
    }
    
    div(myTan, y, x);
    if(!myTan.i.s)
        myTan.i.s = true;
    atan(z, myTan);
    
    if(!x.i.s && y.i.s)
    {
        sub(z, myPi, z);
        return;
    }
    
    if(!x.i.s && !y.i.s)
    {
        sub(z, z, myPi);
        return;
    }
    
    if(x.i.s && !y.i.s)
    {
        if(z.i.s)
            z.i.s = false;
        else
            z.i.s = true;
    }
    
}/* atan2 */

fp atan2(const fp& y, const fp& x)
{
    fp        z;
    
    atan2(z, y, x);
    return z;
}/* atan2 */

void asin(fp& z, const fp& x)
{
    fp                  one=1;
    fp                  theTan, y, temp;
    bool                isNegative;
    
    if(x.i==0)
    {
        z = 0;
        return;
    }
    
    temp = x;
    if(!temp.i.s)
    {
        temp.i.s = true;
        isNegative = true;
    }
    else
        isNegative = false;
    
    if(!compare(temp, one))
    {
        ComputePi(y);
        y.e--; // pi/2
        if(isNegative)
            y.i.s = false;
        z = y;
        return;
    }
   
    mul(theTan, temp, temp);
    sub(theTan, one, theTan);
    sqrt(theTan, theTan);
    div(theTan, temp, theTan);
    atan(y, theTan);
    
    if(isNegative)
        y.i.s = false;
    
    z = y;
    
}/* asin */


fp asin(const fp& x)
{
    fp        z;
    
    asin(z, x);
    return z;
    
}/* asin */

void acos(fp& z, const fp& x)
{
    fp                  one=1;
    fp                  theTan, y, pi, temp;
    bool            isNegative;
    
    if(x.i==0)
    {
        ComputePi(z);
        z.e--; // pi/2
        return;
    }
    
    temp = x;
    if(!temp.i.s)
    {
        temp.i.s = true;
        isNegative = true;
    }
    else
        isNegative = false;
    
    mul(theTan, temp, temp);
    sub(theTan, one, theTan);
    sqrt(theTan, theTan);
    div(theTan, theTan, temp);
    
    atan(y, theTan);
    
    if(isNegative)
    {
        ComputePi(pi);
        sub(y, pi, y);
    }
    
    z = y;
    
}/* acos */


fp acos(const fp& x)
{
    fp        z;
    
    acos(z, x);
    return z;
    
}/* acos */

void sinh(fp& z, const fp& x)
{
    fp                  pointOne=".1", one=1, two=2;
    fp                  xt, zt, temp, temp1, t2, s, s1, t;
    long                i;
    
    xt = x;
    if(!xt.i.s)
        xt.i.s = true;
    if(compare(xt, pointOne)==1)
    {
        exp(temp, xt);
        div(temp1, one, temp);
        sub(temp1, temp, temp1);
        div(zt, temp1, two);
        z = zt;
        return;
    }
    
    // now abs(x)<=.1 so use power series
    s = 0;
    t = one;
    mul(t2, xt, xt);
    
    for (i=2; ;i+=2)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t, t, t2);
        div(t, t, i*(i+1));
    }
    mul(zt, xt, s);
    z = zt;
    
}/* sinh */

fp sinh(const fp& x)
{
    fp        z;
    
    sinh(z, x);
    return z;
    
}/* sinh */

void cosh(fp& z, const fp& x)
{
    fp            temp, one=1, two=2;
    
    temp = x;;
    exp(temp, temp);
    z = temp;
    div(temp, one, temp);
    add(z, z, temp);
    div(z, z, two);
    
}/* cosh */

fp cosh(const fp& x)
{
    fp        z;
    
    cosh(z, x);
    return z;
    
}/* cosh */

void tanh(fp& z, const fp& x)
{
    fp                temp, temp1;
    
    sinh(temp, x);
    cosh(temp1, x);
    div(z, temp, temp1);
    
}/* tanh */


fp tanh(const fp& x)
{
    fp        z;
    
    tanh(z, x);
    return z;
    
}/* tanh */

void asinh(fp& z, const fp& x)
{
    fp                  pointOne=".1", max="1e30000", one=1, two=2;
    fp                  temp, xt, zt, t2, s, s1, t;
    long                i;
    
    xt = x;
    if(!xt.i.s)
        xt.i.s = true;
    if(compare(xt, max)==1)
    {
        mul(temp, xt, two);
        log(zt, temp);
        if(!x.i.s)
            zt.i.s = false;
        z = zt;
        return;
    }
    
    if(compare(xt, pointOne)==1)
    {
        mul(temp, xt, xt);
        add(temp, temp, one);
        sqrt(temp, temp);
        add(temp, xt, temp);
        log(zt, temp);
        if(!x.i.s)
            zt.i.s = false;
        z = zt;
        return;
    }
    
    // abs(x)<=.1 so use power series
    s = 0;
    t = one;
    mul(t2, x, x);
    t2.i.s = false;
    
    for(i=2; ;i+=2)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t, t, t2);
        mul(t, t, (i-1)*(i-1));
        div(t, t, i*(i+1));
    }
    mul(z, x, s);
    
}/* asinh */

fp asinh(const fp& x)
{
    fp        z;
    
    asinh(z, x);
    return z;
    
}/* asinh */

void acosh(fp& z, const fp& x)
{
    fp                  max="1e30000", one=1, two=2;
    fp                  temp;
    
    if(compare(x, one)==-1) // cosh >= 1 !
    {
        z = -1;
        return;
    }
    
    if(!compare(x, one))
    {
        z = 0;
        return;
    }
    
    if(compare(x, max)==1)
    {
        mul(temp, x, two);
        log(z, temp);
        return;
    }

    mul(temp, x, x);
    sub(temp, temp, one);
    sqrt(temp, temp);
    add(temp, x, temp);
    log(z, temp);
    
}/* acosh */

fp acosh(const fp& x)
{
    fp        z;
    
    acosh(z, x);
    return z;
    
}/* acosh */

void atanh(fp& z, const fp& x)
{
    fp                  one=1, two=2, pointOne=".1";
    fp                  xt, temp, temp1, t2, t3, s, s1, t;
    long                i, result;
    
    xt = x;
    if(!xt.i.s)
        xt.i.s = true;
    result = compare(xt, one);
    if(result>=0) // abs(tanh)<1
    {
        z = -1;
        return;
    }
    
    result = compare(xt, pointOne);
    if(result>=0)  // xt>=.1
    {
        add(temp, one, xt);
        sub(temp1, one, xt);
        div(temp, temp, temp1);
        log(temp, temp);
        div(z, temp, two);
        if(!x.i.s)
            z.i.s = false;
        return;
    }
    
    // now xt<.1 so use power series
    
    s = 0;
    t = one;
    mul(t2, xt, xt);
    t3 = one;
    
    for(i=3; ;i+=2)
    {
        add(s1, s, t);
        if(!compare(s, s1))
            break;
        s = s1;
        mul(t3, t3, t2);
        div(t, t3, i);
    }
    mul(z, xt, s);
    if(!x.i.s)
        z.i.s = false;
    
}/* atanh */

fp atanh(const fp& x)
{
    fp        z;
    
    atanh(z, x);
    return z;
    
}/* atanh */
