/*
 *  polyMath.cpp
 *  fp
 *
 *  Created by Robert Delaney on 5/17/13.
 *  Copyright 2013 Bob Delaney's Science Software. All rights reserved.
 *
 */

#include "matq.h"
#include "matqMath.h"
#include "poly.h"
#include "polyMath.h"
#include "polyConv.h"
#include "mb.h"
#include "mbMod.h"
#include "mbMath.h"
#include "fpComplex.h"
#include "findPolyRoots.h"

extern int decPrec;

void init(poly& z)
{
    z.deg = -1;
    z.array = NULL;
    
}/* init */

// don't use with z.array != 0
void init(poly& z, INT32 deg)
{
    INT32   i;
    
    if(deg<0)
    {
        z.deg = -1;
        z.array = NULL;
        return;
    }
    
    z.deg = deg;
    z.array = (bf*)malloc((z.deg+1)*sizeof(bf));
    for(i=0;i<=z.deg;++i)
    {
        z.array[i].num.n = z.array[i].num.nn = 0;
        z.array[i].den.n = z.array[i].den.nn = 0;
        z.array[i].num.b = NULL;
        z.array[i].den.b = NULL;
    }
    
}/* init */

void myFree(poly& x)
{
    INT32   i;
    
    if(x.array==NULL)
        return;
    
    if(x.deg>=0)
    {
        for(i=0;i<=x.deg;++i)
        {
            free(x.array[i].num.b);
            x.array[i].num.b = NULL;
            free(x.array[i].den.b);
            x.array[i].den.b = NULL;
        }
    }
    
}/* myFree */

void myFree(poly*& x, INT32 numPolys)
{
    INT32   i;
    
    for(i=0;i<numPolys;++i)
        myFree(x[i]);
    
    free(x);
    
}/* myFree */

// caller has lowered x.deg, so create new x and copy from old x
void polyAdjust(poly& x)
{
    poly    xt;
    INT32   i;
    
    init(xt, x.deg);
    for(i=0;i<=xt.deg;++i)
        xt.array[i] = x.array[i];
    
    x = xt;
    
}/* polyAdjust */

// remove zeroes continuos from high order down
void polyNormalize(poly& z, const poly& x)
{
	poly		zt;
    int         deg;
    
    zt = x;
	
    if(x.array==NULL)
    {
        z = zt;
        return;
    }
    
    if(x.array[x.deg].num!=0)
    {
        z = zt;
        return;
    }
    
    deg = zt.deg;
    
	while(!zt.array[zt.deg].num && zt.deg>0)
		zt.deg--;
    
    z = zt;
    zt.deg = deg; // for free(zt)
	
}/* polyNormalize */

poly polyNormalize(const poly& x)
{
    poly    z;
    
    polyNormalize(z, x);
    
    return z;
    
}/* polyNormalize */


// remove low order zeroes, effectively factoring out powers of x in poly(x)
poly polyReduce(const poly& x)
{
	INT32		i, deg;
	poly		z, zt;
	
    deg = x.deg;
	zt = x;
	while(!zt.array[0].num.n && zt.deg>0)
	{
		for(i=0;i<zt.deg;i++)
			zt.array[i] = zt.array[i+1];
		
		zt.deg--;
    }
    z = zt;
    zt.deg = deg; // so zt memory is fully removed
        
	return z;
	
}/* polyReduce */


// converts to integer coefficients
void polyOverZ(poly& z, const poly& x)
{
	mb			c;
	INT32		i;	
	poly		zt;
	
	zt = x;
	
	// get products of all denometers
	c = 1;
	
	for(i=0;i<=zt.deg;i++)
		c = c*zt.array[i].den;
	
	// convert to integers
	for(i=0;i<=zt.deg;i++)
		zt.array[i] = c*zt.array[i];
	
	// factor out largest common integer
	if(!zt.deg && !zt.array[0])
	{
		z = zt;
		return;
	}
	
	if(!zt.deg)
	{
		zt.array[0] = 1;
		z = zt;
		return;
	}
	
	if(zt.deg==1)
	{
		c = gcd(zt.array[0].num, zt.array[1].num);
		if(zt.array[1]>0)
		{
			zt.array[0] = zt.array[0] / c;
			zt.array[1] = zt.array[1] / c;
		}
		else
		{
			zt.array[0] = -zt.array[0] / c;
			zt.array[1] = -zt.array[1] / c;
		}
		z = zt;
		return;
	}
	
	// now zt.deg>1
	c = gcd(zt.array[0].num, zt.array[1].num);
	
	if(c==1)
	{
		z = zt;
		return;
	}
	
	for(i=2;i<=zt.deg;i++)
	{
		c = gcd(c, zt.array[i].num);
		if(c==1)
			break;
	}
	
	if(c>1)
	{
		if(zt.array[zt.deg]>0)
		{
			for(i=0;i<=zt.deg;i++)
				zt.array[i] = zt.array[i] / c;
		}
		else
		{
			for(i=0;i<=zt.deg;i++)
				zt.array[i] = -zt.array[i] / c;
		}
	}
		
	z = zt;
	return;
	   
}/* polyOverZ */


poly polyOverZ(const poly& x)
{
	poly		z;
	
	polyOverZ(z, x);
	
	return z;
	
}/* polyOverZ */

/*
// polyNormalize, find lcd of den's and multiply poly with that, then find gcd of num's and divide by that
void polyFactorDown(poly& z, const poly& x)
{
    poly    zt;
    
    
}*//* polyFactorDown */

void polyDeriv(poly& z, const poly& x)
{
    INT32        i;
    
    if(!x.deg)
    {
        z = 0;
        return;
    }
    
    init(z, x.deg-1);
    
    for(i=1;i<=x.deg;i++)
        z.array[i-1] = x.array[i]*i;
    
    z = polyNormalize(z);
    
}/* polyDeriv */


// takes derivative of poly x
poly polyDeriv(const poly& x)
{
	poly		z;
	
    polyDeriv(z, x);
    
	return z;
	
}/* polyDeriv */

// for input x calculates value of polynomial f
void polyEval(bf& z, const poly& f, const bf& x)
{
    int     i;
    
    z = f.array[f.deg];
    for(i=f.deg-1;i>=0;i--)
        z = z * x + f.array[i];
    
}/* polyEval */

bf polyEval(const poly& f, const bf& x)
{
    bf  z;
    polyEval(z, f, x);
    return z;
    
}/* polyEval */

// returns 1 if x=y, 0 if x!=y
int compare(const poly& x, const poly& y)
{
	poly		z;
	
	z = x - y;
	
	if(!z.deg && !z.array[0])
		return 1;
	
	return 0;
	
}/* compare */


void add(poly& z, const poly& x, const poly& y)
{
	INT32		i;
	poly		zt;
	
    init(zt, max(x.deg, y.deg));
	
	if(x.deg>=y.deg)
	{
		for(i=0;i<=y.deg;i++)
			zt.array[i] = x.array[i] + y.array[i];
		
		for(i=y.deg+1;i<=x.deg;i++)
			zt.array[i] = x.array[i];
	}
	else
	{
		for(i=0;i<=x.deg;i++)
			zt.array[i] = x.array[i] + y.array[i];
		
		for(i=x.deg+1;i<=y.deg;i++)
			zt.array[i] = y.array[i];
	}
	
	zt = polyNormalize(zt);
	
	z = zt;

}/* add */


void sub(poly& z, const poly& x, const poly& y)
{
	INT32		i;
	poly		zt;
	
    init(zt, max(x.deg, y.deg));
	
	if(x.deg>=y.deg)
	{
		for(i=0;i<=y.deg;i++)
			zt.array[i] = x.array[i] - y.array[i];
		
		for(i=y.deg+1;i<=x.deg;i++)
			zt.array[i] = x.array[i];
	}
	else
	{
		for(i=0;i<=x.deg;i++)
			zt.array[i] = x.array[i] - y.array[i];
		
		for(i=x.deg+1;i<=y.deg;i++)
			zt.array[i] = -y.array[i];
	}
	
    zt = polyNormalize(zt);
	
	z = zt;
	
}/* sub */


void mul(poly& z, const poly& x, const poly& y)
{
	INT32		i, j;
	poly		zt;
	
	//init(zt, max(x.deg, y.deg));
    init(zt, x.deg+y.deg);
    
	for(i=0;i<=zt.deg;i++)
		zt.array[i] = 0;
	
	for(i=0;i<=x.deg;i++)
		for(j=0;j<=y.deg;j++)
			zt.array[i+j] = zt.array[i+j] + x.array[i]*y.array[j];
	
	zt = polyNormalize(zt);
	
	z = zt;

}/* mul */

static poly singleTerm(INT32 deg, bf c) // can't use bf& c
{
    INT32       i;
    poly        z;
    
    init(z, deg);
    z.array[z.deg] = c;
    for(i=0;i<z.deg;++i)
        z.array[i] = 0;
    
    return z;
    
}/* singleTerm */

// https://en.wikipedia.org/wiki/Polynomial_greatest_common_divisor
// want q = x/y with remainder r so x = q*y + r
void divEuclid(poly& q, poly& r, const poly& x, const poly& y)
{
    poly        xt, yt, qt, rt, d, qTerm;
    bf          c;
    
    xt = x;
    yt = y;
    
    if(!yt.deg)
    {
        if(!yt.array[0])
        {
            q = 0;
            r = 0;
            return;
        }
        qt = xt / yt.array[0];
        r = 0;
        q = qt;
        return;
    }
    
    if(xt==yt)
    {
        qt = 1;
        rt = 0;
        q = qt;
        r = rt;
        return;
    }
    
    if(xt.deg<yt.deg)
    {
        // qt=0 and rt = x
        init(qt, 0);
        qt.array[0] = 0;
        rt = xt;
        q = qt;
        r = rt;
        return;
    }
    
    rt = xt; // init as dividend
    d = yt;  // init as divisor
    qt = 0;  // init as quotient;
    
    c = 1 / d.array[d.deg];
    
    while(rt.deg>=d.deg)
    {
        qTerm = c*rt.array[rt.deg]*xTon(rt.deg-d.deg);
        qt = qt + qTerm;
        rt = rt - qTerm * d;
        rt = polyNormalize(rt);
    }
    q = qt;
    r = rt;
    
}/* divEuclid */
	

// this is only for monic integer polynomials and for prime p!
// replace division by inverse mod
// want q = x/y with remainder r so x = q*y + r mod p
void divEuclid(poly& q, poly& r, const poly& x, const poly& y, const mb& p)
{
    poly        xt, yt, qt, rt, d, qTerm;
    bf          c;
    
    xt = x % p;
    yt = y % p;
    
    xt = polyNormalize(xt);
    yt = polyNormalize(yt);
    
    if(!yt.deg)
    {
        if(!yt.array[0])
        {
            q = 0;
            r = 0;
            return;
        }
        c = yt.array[0];
        invMod(c.num, c.num, p);
        c = c.num;
        qt = xt * c;
        r = 0;
        q = qt;
        return;
    }
    
    if(xt==yt)
    {
        qt = 1;
        rt = 0;
        q = qt;
        r = rt;
        return;
    }
    
    if(xt.deg<yt.deg)
    {
        // qt=0 and rt = x
        init(qt, 0);
        qt.array[0] = 0;
        rt = xt;
        q = qt;
        r = rt;
        return;
    }
    
    rt = xt; // init as dividend
    d = yt;  // init as divisor
    qt = 0;  // init as quotient;
    
    c = d.array[d.deg];
    invMod(c.num, c.num, p);
    c = c.num;
    
    while(rt.deg>=d.deg)
    {
        qTerm = c*rt.array[rt.deg]*xTon(rt.deg-d.deg) % p;
        qt = qt + qTerm;
        rt = (rt - qTerm * d) % p;
        rt = polyNormalize(rt);
    }
    q = qt;
    r = rt;
	
}/* divEuclid */

// https://en.wikipedia.org/wiki/Polynomial_greatest_common_divisor#Euclidean_division
// this is only for monic integer polynomials!
// replace division by inverse mod
// want q = x/y with remainder r so x = q*y + r mod p
void divEuclid(poly& q, poly& r, const poly& x, const poly& y, double p)
{
    poly        xt, yt, qt, rt, d, qTerm;
    bf          c;

    xt = x % p;
    yt = y % p;
    
    xt = polyNormalize(xt);
    yt = polyNormalize(yt);
    
    if(!yt.deg)
    {
        if(!yt.array[0])
        {
            q = 0;
            r = 0;
            return;
        }
        c = yt.array[0];
        invMod(c.num, c.num, p);
        c = c.num;
        qt = xt * c;
        r = 0;
        q = qt;
        return;
    }
    
    if(xt==yt)
    {
        qt = 1;
        rt = 0;
        q = qt;
        r = rt;
        return;
    }
    
    if(xt.deg<yt.deg)
    {
        // qt=0 and rt = x
        init(qt, 0);
        qt.array[0] = 0;
        rt = xt;
        q = qt;
        r = rt;
        return;
    }
    
    rt = xt; // init as dividend
    d = yt;  // init as divisor
    qt = 0;  // init as quotient;
    
    c = d.array[d.deg];
    invMod(c.num, c.num, p);
    c = c.num;
    
    while(rt.deg>=d.deg)
    {
        qTerm = c*rt.array[rt.deg]*xTon(rt.deg-d.deg) % p;
        qt = qt + qTerm;
        rt = (rt - qTerm * d) % p;
        rt = polyNormalize(rt);
    }
    q = qt;
    r = rt;
    
}/* divEuclid */


void div(poly& z, const poly& x, const poly& y)
{
	poly		q, r;
	
	divEuclid(q, r, x, y);
	
	z = q;

}/* div */

void div(poly& z, const poly& x, const poly& y, double p)
{
    poly        q, r;
    
    divEuclid(q, r, x, y, p);
    
    z = q;
    
}/* div */

void add(poly& z, const poly& x, const bf& y)
{
	poly		zt;
	
	zt = x;
	zt.array[0] = zt.array[0] + y;
	
	z = zt;
}


void sub(poly& z, const poly& x, const bf& y)
{
	poly		zt;
	
	zt = x;
	zt.array[0] = zt.array[0] - y;
	
	z = zt;
}


void mul(poly& z, const poly& x, const bf& y)
{
	INT32		i;
	poly		zt;
	
	zt = x;
	for(i=0;i<=zt.deg;i++)
		zt.array[i] = y*zt.array[i];
	
	zt = polyNormalize(zt);  // for y=0
	
	z = zt;
}


void div(poly& z, const poly& x, const bf& y)
{
	INT32		i;
	poly		zt;
	
	if(!y)
	{
		z = 0;
		return;
	}
	
	zt = x;
	for(i=0;i<=zt.deg;i++)
		zt.array[i] = zt.array[i] / y;
	
	z = zt;
}


void add(poly& z, const bf& x, const poly& y)
{
	add(z, y, x);
}


void sub(poly& z, const bf& x, const poly& y)
{
	sub(z, y, x);
	z = -z;
}


void mul(poly& z, const bf& x, const poly& y)
{
	mul(z, y, x);
}


void div(poly& z, const bf& x, const poly& y)
{
	z.deg = 0;
    init(z, z.deg);
	if(y.deg>0)
	{
		z.array[0] = 0;
		return;
	}
	
	// now y.deg = 0
	if(!y.array[0])
	{
		z.array[0] = 0;
		return;
	}
	
	z.array[0] = x / y.array[0];
}/* div */

// uses lcm to find lcd of den's
mb lcd(const poly& x)
{
    mb      z;
    INT32   i;
    
    z = 1;
    for(i=0;i<=x.deg;++i)
        z = lcm(z, x.array[i].den);
    return z;
    
}/* lcd */

void gcd(poly& z, const poly& x, const poly& y)
{
	poly		q, r, xt, yt, zt, qt, rt;
	
	
	xt = x;
	yt = y;
    
    if(!xt.array[0] && yt.array[0]!=0)
    {
        // x cannot be in the gcd
        xt = polyReduce(xt);
    }
    
    if(xt.array[0]!=0 && !yt.array[0])
    {
        // x cannot be in the gcd
        yt = polyReduce(yt);
    }
    
    if(xt==yt)
    {
        z = xt;
        return;
    }
	
	if(yt.deg>xt.deg)
	{
		r = xt;
		xt = yt;
		yt = r;
	}
    
    // now xt.deg>=yt.deg
    
    divEuclid(q, r, xt, yt);
    if(r==0)
    {
        z = yt;
        return;
    }
    
	while(yt.deg>0)
	{
		divEuclid(q, r, xt, yt);  // xt = q*yt + r
		xt = yt;
		yt = r;
	}
    
    if(!yt.array[0].num)
    {
        z = xt;
        return;
    }
    
    // gcd = 1
    init(zt, 0);
    zt.array[0] = 1;
    z = zt;
	
}/* gcd */

poly gcd(const poly& x, const poly& y)
{
    poly    z;
    
    gcd(z, x, y);
    
    return z;
    
}/* gcd */

// only for integer polynomials, p must be prime
void makeMonic(poly& z, const poly& x, double p)
{
    poly    zt;
    bf      temp;
    mb      tempmb;
    
    zt = x % p;
    temp = zt.array[zt.deg];
    if(temp==1)
    {
        z = zt;
        return;
    }
    
    //invMod(temp.num, temp.num, p); // never do this
    //temp = temp.num;
    invMod(tempmb, temp.num, p);
    temp = tempmb;
    zt = (zt * temp) % p;
    z = zt;
   
}/* makeMonic */

// only for integer polynomials, p must be prime
void makeMonic(poly& z, const poly& x, const mb& p)
{
    bf      temp;
    mb      tempmb;
    
    temp = x.array[x.deg];
    if(temp==1)
        return;
    
    //invMod(temp.num, temp.num, p); // never do this
    //temp = temp.num;
    invMod(tempmb, temp.num, p);
    temp = tempmb;
    z = (x * temp) % p;
    
}/* makeMonic */

void xTon(poly& z, int n)
{
    int     i;
    
    init(z, n);
    z.array[z.deg] = 1;
    for(i=0;i<z.deg;++i)
        z.array[i] = 0;
    
}/* xTon */


poly xTon(int n)
{
    poly     z;
    
    xTon(z, n);
    
    return z;
    
}/* xTon */

// https://en.wikipedia.org/wiki/Polynomial_greatest_common_divisor
// for monic polynomials mod p
// if g.deg>=f.deg, gcd(f(x), g(x)) = gcd(f(x), g(x) - x^n*f(x)) n = g.deg-f.deg
void gcd(poly& z, const poly& x, const poly& y, double p)
{
    poly        q, r, xt, yt, zt, qt, rt;
    
    
    xt = x % p;
    yt = y % p;
    
    if(!xt.array[0] && yt.array[0]!=0)
    {
        // x cannot be in the gcd
        xt = polyReduce(xt);
    }
    
    if(xt.array[0]!=0 && !yt.array[0])
    {
        // x cannot be in the gcd
        yt = polyReduce(yt);
    }
    
    if(xt==yt)
    {
        z = xt;
        return;
    }
    
    if(yt.deg>xt.deg)
    {
        r = xt;
        xt = yt;
        yt = r;
    }
    
    // now xt.deg>=yt.deg
    divEuclid(q, r, xt, yt, p);
    if(r==0)
    {
        z = yt;
        return;
    }
    
    while(yt.deg>0)
    {
        divEuclid(q, r, xt, yt, p);  // xt = q*yt + r
        xt = yt;
        yt = r;
        if(yt.array[yt.deg]!=1)
            makeMonic(yt, yt, p);
    }
    
    if(!yt.array[0].num)
    {
        z = xt;
        return;
    }
    
    // gcd = 1
    init(zt, 0);
    zt.array[0] = 1;
    z = zt;
    
}/* gcd */

poly gcd(const poly& x, const poly& y, double p)
{
    poly    z;
    
    gcd(z, x, y, p);
    return z;
    
}/* gcd */

// https://en.wikipedia.org/wiki/Polynomial_greatest_common_divisor
// for monic polynomials mod p
// if g.deg>=f.deg, gcd(f(x), g(x)) = gcd(f(x), g(x) - x^n*f(x)) n = g.deg-f.deg
void gcd(poly& z, const poly& f, const poly& g, const mb& p)
{
    poly    temp, ft, gt, zt, qt, rt;
    
    ft = f % p;
    gt = g % p;
    
    if(!ft.array[0] && gt.array[0]!=0)
    {
        // x cannot be in the gcd
        ft = polyReduce(ft);
    }
    
    if(ft.array[0]!=0 && !gt.array[0])
    {
        // x cannot be in the gcd
        gt = polyReduce(gt);
    }
    
    makeMonic(ft, ft, p);
    makeMonic(gt, gt, p);
    
    if(!ft.deg || !gt.deg || f==0 || g==0)
    {
        init(zt, 0);
        zt.array[0] = 1;
        z = zt;
        return;
    }
    
    if(gt.deg<ft.deg)
    {
        temp = gt;
        gt = ft;
        ft = temp;
    }
    
    // now ft.deg>=gt.deg
    divEuclid(qt, rt, ft, gt, p);
    if(rt==0)
    {
        z = gt;
        return;
    }
    
    while(ft != gt && ft != 0)
    {
        gt = gt - ft*xTon(gt.deg-ft.deg);
        makeMonic(gt, gt, p);
        gt = gt % p;
        if(gt.deg<ft.deg)
        {
            temp = gt;
            gt = ft;
            ft = temp;
        }
    }
    z = ft;
    
}/* gcd */

poly gcd(const poly& f, const poly& g, const mb& p)
{
    poly    z;
    
    gcd(z, f, g, p);
    
    return z;
    
}/* gcd */

// it is assumed that x, y already belong to GF(q)[x]
// and that if n = highest coefficient of effective dividing polynomial in divEuclid then gcd(n, q) = 1
poly gcdYun(const poly& x, const poly& y, double q)
{
    poly        p, r, xt, yt;
    bool        hasFactorx;
    
    xt = x % q;
    yt = y % q;
    
    xt = polyReduce(xt);
    yt = polyReduce(yt);
    
    
    hasFactorx = false;
    if(xt.deg<x.deg && yt.deg<y.deg)
        hasFactorx = true;
    
    if(yt.deg>xt.deg)
    {
        r = xt;
        xt = yt;
        yt = r;
    }
    
    while(!(!yt.deg && !yt.array[0]))
    {
        divEuclid(p, r, xt, yt, q);  // xt = q*yt + r mod q
        xt = yt % q;
        yt = r % q;
        if(yt.deg>xt.deg)
        {
            r = xt;
            xt = yt;
            yt = r;
        }
    }
    
    xt = polyOverZ(xt);
    
    if(xt.deg==0 && hasFactorx)
    {
        myFree(xt);
        init(xt, 1);
        xt.array[0] = 0;
        xt.array[1] = 1;
    }
    
    return xt;
    
}/* gcdYun */


// with x = q*y + r, r = x mod y
void polyModPoly(poly& z, const poly& x, const poly& y)
{
	poly		q, r;
	
	divEuclid(q, r, x, y);
	
	//r = polyOverZ(r);
    
    z = r;
	
}/* polyModPoly */

poly polyModPoly(const poly& x, const poly& y)
{
    poly    z;
    
    polyModPoly(z, x, y);
    
    return z;
    
}/* polyModPoly */

// with x = q*y + r, r = x mod y mod q
void polyModPolyModq(poly& z, const poly& x, const poly& y, double q)
{
	poly		p, r;
	
	divEuclid(p, r, x, y, q);
	
	//r = polyOverZ(r);
	
	//z = polyMod(r, q);
	
	r = polyNormalize(r); // mod might make high order coefficient zero
	
    z = r;
	
}/* polyModPolyModq */

poly polyModPolyModq(const poly& x, const poly& y, double q)
{
    poly    z;
    
    polyModPolyModq(z, x, y, q);
    
    return z;
    
}/* polyModPolyModq */

// with x = q*y + r, r = x mod y mod q
void polyModPolyModq(poly& z, const poly& x, const poly& y, const mb& q)
{
    poly        p, r;
    
    divEuclid(p, r, x, y, q);
    
    //r = polyOverZ(r);
    
    //z = polyMod(r, q);
    
    r = polyNormalize(r); // mod might make high order coefficient zero
    
    z = r;
    
}/* polyModPolyModq */

poly polyModPolyModq(const poly& x, const poly& y, const mb& q)
{
    poly    z;
    
    polyModPolyModq(z, x, y, q);
    
    return z;
}/* polyModPolyModq */

// if poly is not integer, multiply by lcd and then take mod
// reduces each coefficient of x by taking mod q
poly polyMod(const poly& x, const mb& q)
{
	poly		xt;
	INT32		i;
    bf          xlcd;
	
	xt = x;
    if(!isPolyInteger(xt))
    {
        xlcd = lcd(xt);
        xt = xt*xlcd;
    }
	
	for(i=0;i<=xt.deg;i++)
		xt.array[i].num = xt.array[i].num % q;
    
    xt = polyNormalize(xt);
	
	return xt;
	
  }/* polyMod */

// f poly is not integer, multiply by lcd and then take mod
void polyMod(poly& z, const poly& x, double q)
{
	poly		zt;
	INT32		i;
	bf          zlcd;
    
	zt = x;
    if(!isPolyInteger(zt))
    {
        zlcd = lcd(zt);
        zt = zt*zlcd;
    }
	
	for(i=0;i<=zt.deg;i++)
		zt.array[i].num = zt.array[i].num % q;
	
    z = polyNormalize(zt);
	
}/* polyMod */

poly polyMod(const poly& x, double q)
{
    poly    z;
    
    polyMod(z, x, q);
    
    return z;
    
}/* polyMod */

// z = x^p
void polyPow(poly& z, const poly& x, INT32 p)
{
	INT32				i, n;
	static UINT32		mask; // 0x80000000
	poly				xt, zt;
	static bool			initGood=false;
	
	if(!initGood)
	{
		mask = 1;
		mask = (mask<<31); // 0x80000000
		initGood = true;
	}
	
	//zt = x;
	
	if(p<0)
	{
		zt.deg = 0;
        init(zt, zt.deg);
		zt.array[0] = 0;
		z = zt;
		return;
	}
	
	if(!p)
	{
		zt.deg = 0;
        init(zt, zt.deg);
		zt.array[0] = 1;
		z = zt;
		return;
	}
    
	// now p>0
	//xt = polyOverZ(x);

	zt = 1;
	n = NumBits(p);
	
	p = (p<<(32-n));  // go to leading 1 bit
	
	for(i=0;i<n;i++)
	{
		mul(zt, zt, zt); // zt^2
		if(p & mask)
			mul(zt, zt, x);
		p = (p<<1);
	}
	
	z = zt;
	
}/* polyPow */


poly  polyPow(const poly& x, INT32 p)
{
	poly		z;
	
	polyPow(z, x, p);
	
	return z;
	
}/* polyPow */

void polyPowModq(poly& z, const poly& x, INT32 p, double q)
{
    INT32                   i, n;
    static UINT32           mask; // 0x80000000
    poly                    xt, zt;
    static bool             initGood=false;
    
    if(!initGood)
    {
        mask = 1;
        mask = (mask<<31); // 0x80000000
        initGood = true;
    }
    
    //zt = x;
    
    if(p<0)
    {
        zt.deg = 0;
        init(zt, zt.deg);
        zt.array[0] = 0;
        z = zt;
        return;
    }
    
    if(!p)
    {
        zt.deg = 0;
        init(zt, zt.deg);
        zt.array[0] = 1;
        z = zt;
        return;
    }
    
    if(!x.deg)
    {
        zt.deg = x.deg;
        init(zt, zt.deg);
        power(zt.array[0], zt.array[0], p);
        z = zt % q;
        return;
    }
    
    // now p>0
    //xt = polyOverZ(x);
    
    zt = 1;
    n = NumBits(p);
    
    p = (p<<(32-n));  // go to leading 1 bit
    
    for(i=0;i<n;i++)
    {
        mul(zt, zt, zt); // zt^2
        zt = zt % q;
        if(p & mask)
            mul(zt, zt, x);
        zt = zt % q;
        p = (p<<1);
    }
    zt = polyNormalize(zt);
    z = zt;
    
}/* polyPowModq */

poly polyPowModq(const poly& x, INT32 p, double q)
{
    poly    z;
    
    polyPowModq(z, x, p, q);
    
    return z;
    
}/* polyPowModq */

// z = x^p mod y
void polyPowerMod(poly& z, const poly& x, INT32 p, const poly& y)
{
	INT32				i, n;
	static UINT32		mask; // 0x80000000
	poly				xt, zt;
	static bool			initGood=false;
	
	if(!initGood)
	{
		mask = 1;
		mask = (mask<<31); // 0x80000000
		initGood = true;
	}
	
	if(p<0)
	{
		z = 0;
		return;
	}
	
	if(!p)
	{
		zt = 1;
		z = polyModPoly(zt, y);
		return;
	}
	
	zt = x;
	
	if(!zt.deg)
	{
		zt.array[0] = power(zt.array[0].num, p);
		z = polyModPoly(zt, y);
		return;
	}
	
	// now p>0
	xt = polyOverZ(x);
	zt = 1;
	n = NumBits(p);
	
	p = (p<<(32-n));  // go to leading 1 bit
	
	for(i=0;i<n;i++)
	{
		mul(zt, zt, zt); // zt^2
		zt = polyModPoly(zt, y);
		if(p & mask)
		{
			mul(zt, zt, xt);
			zt = polyModPoly(zt, y);
		}
		p = (p<<1);
	}
	
	z = zt;
	
}/* polyPowerMod */


// z = x^p mod y mod q
void polyPowerModPolyModq(poly& z, const poly& x, INT32 p, const poly& y, double q)
{
	INT32				i, n;
	static UINT32		mask; // 0x80000000
	poly				xt, zt;
	static bool			initGood=false;
	
	if(!initGood)
	{
		mask = 1;
		mask = (mask<<31); // 0x80000000
		initGood = true;
	}
	
	if(p<0)
	{
		z = 0;
		return;
	}
	
	if(!p)
	{
		zt = 1;
		zt = polyModPoly(zt, y);
		z = polyMod(zt, q);
		return;
	}
	
	zt = x;
	
	if(!zt.deg)
 	{
		zt.array[0] = power(zt.array[0].num, p);
		zt = polyModPoly(zt, y);
		z = polyMod(zt, q);
		return;
	}
	
	// now p>0
	xt = polyOverZ(x);
	zt = 1;
	n = NumBits(p);
	
	p = (p<<(32-n));  // go to leading 1 bit
	
	for(i=0;i<n;i++)
	{
		mul(zt, zt, zt); // zt^2
		zt = polyModPoly(zt, y);
		zt = polyMod(zt, q);
		if(p & mask)
		{
			mul(zt, zt, xt);
			zt = polyModPoly(zt, y);
			zt = polyMod(zt, q);
		}
		p = (p<<1);
	}
	
	z = zt;
	
}/* polyPowerModPolyModq */


poly polyPowerMod(const poly& x, INT32 p, const poly& y)
{
	poly		z;
	
	polyPowerMod(z, x, p, y);
	
	return z;
	
}/* polyPowerMod */


poly polyPowerModPolyModq(const poly& x, INT32 p, const poly& y, double q)
{
	poly		z;
	
	polyPowerModPolyModq(z, x, p, y, q);
	
	return z;
	
}/* polyPowerModPolyModq */

void polyPowerModPoly(poly& z, const poly& x, const mb& p, const poly& y)
{
    poly    zt;
    int     bitIndex;
    
    zt = 1;
    bitIndex = NumBits(p) - 1; // points to highest 1 bit
    while(bitIndex>=0)
    {
        zt = zt * zt;
        polyModPoly(zt, y);
        if(mbGetBit(p, bitIndex--))
        {
            zt = zt * x;
            polyModPoly(zt, y);
        }
    }
}/* polyPowerModPoly */

poly polyPowerModPoly(const poly& x, const mb& p, const poly& y)
{
    poly    z;
    
    polyPowerModPoly(z, x, p, y);
    
    return z;
    
}/* polyPowerModPoly */

void polyPowerModPolyModq(poly& z, const poly& x, const mb& p, const poly& y, const mb& q)
{
    poly    zt;
    int     bitIndex;
    
    zt = 1;
    bitIndex = NumBits(p) - 1; // points to highest 1 bit
    while(bitIndex>=0)
    {
        zt = (zt * zt) % q;
        
        zt = polyModPolyModq(zt, y, q);
        if(mbGetBit(p, bitIndex--))
        {
            zt = (zt * x) % q;
            zt = polyModPolyModq(zt, y, q);
        }
    }
    
}/* polyPowerModPolyModq */

poly polyPowerModPolyModq(const poly& x, const mb& p, const poly& y, const mb& q)
{
    poly    z;
    
    polyPowerModPolyModq(z, x, p, y, q);
    
    return z;
}/* polyPowerModPolyModq */

// x has null vectors as its columns
void convNullVectorsToPolys(poly*& z, const matq& x, double q)
{
    INT32   numNV, deg, i, j;
    
    numNV = x.nc;
    z = (poly*)malloc(numNV*sizeof(poly));
    deg = x.nr - 1; // max degree
    for(i=0;i<numNV;++i)
        init(z[i], deg);
    
        for(j=0;j<numNV;++j)
            for(i=0;i<=deg;++i)
                z[j].array[i] = x.array[i][j];
    
    for(i=0;i<numNV;++i)
    {
        polyNormalize(z[i], z[i]);
    }
    
}/* convNullVectorsToPolys */

// the integer array z will contain the primes which divide n and count gives their number
void primesDivide(int*& z, int& count, int n)
{
    int     i, j;
    mb      temp;
    
    z = (int*)malloc((n/2+1)*sizeof(int));  // overcount of possible primes
    
    j = 0;
    count = 0;
    if(2*(n/2)==n)
    {
        z[j++] = 2;
        count++;
    }
    
    for(i=3;i<=n;i+=2)
    {
        temp = i;
        if(isItPrime(temp, 35))
        {
            if(i*(n/i)==n)
            {
                z[j++] = i;
                count++;
            }
        }
    }
    
}/* primesDivide */

// the integer array z will contain the primes which divide n and count gives their number
void primesDivide(mb*& z, int& count, const mb& n)
{
    int     j, nn;
    mb      i, temp;
    
    equate(nn, n);
    z = (mb*)malloc((nn/2+1)*sizeof(mb));  // overcount of possible primes
    for(j=0;j<nn/2+1;++j)
        init(z[j]);
    
    j = 0;
    count = 0;
    if(2*(n/2)==n)
    {
        z[j++] = 2;
        count++;
    }
    
    for(i=3;i<=n;i+=2)
    {
        if(isItPrime(i, 35))
        {
            if(i*(n/i)==n)
            {
                z[j++] = i;
                count++;
            }
        }
    }
}/* primesDivide */


// Algorithm Rabin Irreducibility Test
// corrects error in article below
// https://en.wikipedia.org/wiki/Factorization_of_polynomials_over_finite_fields#Victor_Shoup.27s_algorithm
// returns true if poly f is reducible over GF(p) p prime, false if irreducible
// f must be monic
bool isPolyReducible(const poly& f, double p)
{
    int     i, n, count, *thePrimes, *nn;
    poly    x, h, g, ft;
    int    theExp;
    
    ft = f;
    ft = polyNormalize(ft);
    makeMonic(ft, ft, p);
    
    if(ft.deg<=1)
        return false;
    
    n = ft.deg;
    
    
    primesDivide(thePrimes, count, n);
    nn = (int*)malloc(count*sizeof(int));
    for(i=0;i<count;++i)
        nn[i] = n / thePrimes[i];
    
    init(x, 1);
    x.array[0] = 0;
    x.array[1] = 1;
    // poly polyModPoly(const poly& x, const poly& y, double q)
    for(i=0;i<count;++i)
    {
        theExp = pow(p, (double)nn[i]);
        //h = polyPowerMod(x, theExp, ft, p);
        h = polyModPolyModq(xTon(theExp)-x, ft, p);
        g = gcd(ft, h, p);
        if(g!=1)
            return true;
    }
    //int  expInt, phiInt;
    // g==1 for all so now have final test
    theExp = pow(p, (double)n);
    //expInt = theExp;
    //phiInt = p - 1;
    //expInt = expInt % phiInt;
    //g = polyPowerMod(x, theExp, ft, p);
    g = polyModPolyModq(xTon(theExp)-x, ft, p);
    if(g==0)
        return false;
    
    return true;
    
}/* isPolyReducible */

bool isPolyReducible(const poly& f, const mb& p)
{
    int     i, count, expInt;
    mb      *thePrimes, *nn, n;
    poly    x, h, g, ft;
    mb      theExp;
    
    if(p<2)
        return false;
    
    ft = f;
    ft = polyNormalize(ft);
    makeMonic(ft, ft, p);
    
    if(ft.deg<=1)
        return false;
    
    n = ft.deg;
    
    primesDivide(thePrimes, count, n);
    
    nn = (mb*)malloc(count*sizeof(mb));
    
    for(i=0;i<count;++i)
    {
        init(nn[i]);
        nn[i] = n / thePrimes[i];
    }
    
    init(x, 1);
    x.array[0] = 0;
    x.array[1] = 1;
    // poly polyModPolyModq(const poly& x, const poly& y, const mb& q)
    
    for(i=0;i<count;++i)
    {
        theExp = power(p, nn[i]);
        equate(expInt, theExp);
        //theExp = theExp % phi(p);
        //h = polyPow(x, theExp);
        //h = polyPowerModPolyModq(x, theExp, ft, p);
        h = polyModPolyModq(xTon(expInt)-x, ft, p);
        g = gcd(ft, h, p);
        if(g!=1)
            return true;
    }
    // g==1 for all so now have final test
    theExp = power(p, n);
    //theExp = theExp % phi(p);
    equate(expInt, theExp);
    //g = polyPowerModPolyModq(x, theExp, ft, p);
    g = polyModPolyModq(xTon(expInt)-x, ft, p);
    if(g==0)
        return false;
    
    return true;
    
}/* isPolyReducible */

// use primitive poly x to generate poly cyclic group of order p^m - 1
void genPolyCyclicGroup(poly*& z, const poly& x, double p)
{
    int     i, m, order;
    poly    temp, temp1, xt;
    
    m = x.deg;
    order = pow(p, (double)m) - 1;
    init(xt, 1);
    xt.array[0] = 0;
    xt.array[1] = 1;
    
    z = (poly*)malloc(order*sizeof(poly));
    for(i=0;i<order;++i)
        init(z[i]);
    
    init(temp, 0);
    temp.array[0] = 1;
    
    for(i=0;i<order;++i)
    {
        if(temp.deg==m)
        {
            temp1 = temp.array[m] * x;
            temp = (temp - temp1) % p;
        }
        //temp1 = polyModPolyModq(temp, x, p);
        z[i] = temp;
        temp = temp*xt;
    }
    
}/* genPolyCyclicGroup */

// use primitive poly x to generate poly cyclic group of order p^m - 1
void genPolyCyclicGroup(char*& zString, const poly& x, double p)
{
    int         i, m, order, length;
    poly        *z, temp;
    char        is[10], *pString;
    
    m = x.deg;
    order = pow(p, (double)m) - 1;
    
    genPolyCyclicGroup(z, x, p);
    
    // estimate length of zString
    
    length = 0;
    for(i=0;i<order;++i)
    {
        polyConvToxString(pString, z[i]);
        length+=strlen(pString);
        free(pString);
    }
    
    // add for x^i = , space, \n so 15*order
    length+=(15*order);
    
    zString = (char*)malloc(length*sizeof(char));
    zString[0] = 0;
    for(i=0;i<order;++i)
    {
        strcat(zString, "x^");
        sprintf(is, "%d", i);
        strcat(zString, is);
        strcat(zString, " = ");
        polyConvToxString(pString, z[i]);
        strcat(zString, pString);
        free(pString);
        strcat(zString, "\n");
    }
    
}/* genPolyCyclicGroup */

// returns true if poly is Imnteger and Monic, else false
bool isPolyIntMonic(const poly& x)
{
    int         i;
    
    if(x.array[x.deg]!=1)
        return false;
    
    for(i=0;i<=x.deg;++i)
        if(x.array[i].den!=1)
            return false;
    
    return true;
    
}/* isPolyIntMonic */

// returns true if poly is Imnteger, else false
bool isPolyInteger(const poly& x)
{
    int         i;
    
    for(i=0;i<=x.deg;++i)
        if(x.array[i].den!=1)
            return false;
    
    return true;
    
}/* isPolyInteger */


// https://en.wikipedia.org/wiki/Finite_field#Applications
// Zech's Logarithms  a^j + a^i = a^i (a^(j-i) + I) ; so just need to verify a^n + I, n = 0, 1, ... , order-1   order = p^m - 1
// belongs to field
// this generates Zech's Logarithms for cyclic poly group based on primitive poly x
// when poly for log is zero, we put -1 for log since zero means x^0 (x here is not prim poly!)
bool polyZechLogs(int*& z, const poly& x, double p)
{
    poly        *pGroup, temp, temp1;
    int         i, j, m, order;
    bool        isThere;
    
    genPolyCyclicGroup(pGroup, x, p);
    m = x.deg;
    order = pow(p, (double)m) - 1;
    z = (int*)malloc(order*sizeof(int));
    
    // load zechLog
    for(i=0;i<order;++i)
    {
        temp = (pGroup[i] + pGroup[0]) % p;
        if(temp.deg==m)
        {
            temp1 = temp.array[m] * x;
            temp = (temp - temp1) % p;
        }
        
        isThere = false;
        for(j=0;j<order;++j)
        {
            if(temp==0)
            {
                z[i] = -1;
                isThere = true;
                break;
            }
            if(temp==pGroup[j])
            {
                z[i] = j;
                isThere = true;
                break;
            }
        }
        if(!isThere)
            return false;
    }
    return true;
    
}/* polyZechLogs */

// for plugin, put Zech's Logarithms into a poly
bool polyZechLogs(poly& z, const poly& x, double p)
{
    int     i, *zInt, m, order;
    poly    zt;
    
    if(!polyZechLogs(zInt, x, p))
        return false;
    
    m = x.deg;
    order = pow(p, (double)m) - 1;
    init(zt, order-1); // holds order values
    for(i=0;i<=zt.deg;++i)
        zt.array[i] = zInt[i];
    
    z = zt;
    
    return true;
    
}/* polyZechLogs */

// uses
// bool findPolyRoots(fpComplex*& root, fpComplex*& Parray, fpComplex*& coeffArray, long n, bool doRoundToZero);
// to convert to polys. Note that x poly has deg n, z array has dimension n as does Parray, coeffArray has dimension n+1.
bool findPolyRoots(fpComplex*& root, fpComplex*& Parray, const poly& x, bool doRoundToZero)
{
    long        i, n;
    fpComplex   *coeffArray;
    
    n = x.deg;
    if(n<1)
        return false;
    
    
    coeffArray = (fpComplex*)malloc((n+1)*sizeof(fpComplex));
    for(i=0;i<=n;++i)
        init(coeffArray[i]);
    
    for(i=0;i<=n;++i)
        coeffArray[i] = x.array[i];
    
    if(!findPolyRoots(root, Parray, coeffArray, n, doRoundToZero))
        return false;
    
    return true;
    
}/* findPolyRoots */

bool findPolyRoots(char*& zString, const poly& x, bool doRoundToZero)
{
    fpComplex       *root, *Parray;
    long            i, n, length;
    
    
    if(!findPolyRoots(root, Parray, x, doRoundToZero))
        return false;
    
    // char* fpComplexToStr(const fpComplex& x, INT32 decPrec);
    // getBlockPrec
    //setBlockPrec(16);
    //decPrec = 16;
    n = x.deg;
    length = 0;
    for(i=0;i<n;++i)
        length = length + strlen(fpComplexToStr(root[i], decPrec)) + strlen(fpComplexToStr(Parray[i], decPrec));
    length = length + 8*n + 100;
    zString = (char*)malloc(length*sizeof(char));
    zString[0] = 0;
    strcpy(zString, "roots:\n");
    
   for(i=0;i<n;++i)
    {
        strcat(zString, fpComplexToStr(root[i], decPrec));
        strcat(zString, "\n\n");
    }
    
    
    strcat(zString, "Polynomial(root):\n");
    for(i=0;i<n;++i)
    {
        strcat(zString, fpComplexToStr(Parray[i], decPrec));
        strcat(zString, "\n\n");
    }
    // free root and Parray here
    for(i=0;i<n;++i)
    {
        myFree(root[i]);
        myFree(Parray[i]);
    }
    
    return true;
    
} /* findPolyRoots */

