//
//  matfpMath.cpp
//  fp
//
//  Created by Robert Delaney on 2/1/17.
//
//

#include "fp.h"
#include "matfpMath.hpp"
#include "matfp.hpp"
#include "matfpConv.hpp"
#include "matc.hpp"
#include "matcConv.hpp"
#include "matcMath.hpp"

static fp abs(fp& x)
{
    fp      z;
    
    z = x;
    if(z<0)
        z.i.n = -z.i.n; // so don't create a new instance
    
    return z;
    
}/* abs */

// swap row i with row j
static void matfpSwapRows(matfp& x, INT32 i, INT32 j)
{
    fp		temp;
    INT32	k;
    
    for(k=0;k<x.nc;k++)
    {
        temp = x.array[i][k];
        x.array[i][k] = x.array[j][k];
        x.array[j][k] = temp;
    }
    
}/* matqSwapRows */


static bool doGauss(matfp& x, INT32& swapParity, INT32 ii, INT32 jj)
{
    fp          temp, temp1;
    INT32       i, j, k;
    bool        isZero;
    
    if(ii<0 || jj<0 || ii>=x.nr || jj>=x.nc)
        return false;
    
    k = 0;
    
    // is pivot zero?
    if(!x.array[ii][jj])
    {
        // must swap rows
        // find non-zero element below pivot in pivot column
        isZero = true;
        for(i=ii+1;i<x.nr;++i)
        {
            if(x.array[i][jj]!=0)
            {
                isZero = false;
                k = i;
                break;
            }
        }
        
        if(isZero)
            return false;
        
        // have non-zero element in row k, so swap rows ii with k
        matfpSwapRows(x, ii, k);
        swapParity = -swapParity;
    }
    
    // now we do the Gauss Elim. operation
    // pivot element xt[ii][jj] is not zero
    
    // now for every k not ii subtract row ii multiplied by xt[k][jj] from row k provided x[k][jj] != 0
    for(k=0;k<x.nr;++k)
    {
        if(k!=ii && (x.array[k][jj]!=0))
        {
            //temp = x[k][jj] / x[ii][jj];
            //DivCC(temp, x[k][jj], x[ii][jj]);
            temp = x.array[k][jj] / x.array[ii][jj];
            for(j=0;j<x.nc;++j)
            {
                //MulCC(temp1, x[ii][j], temp);
                temp1 = x.array[ii][j] * temp;
                //SubCC(x[k][j], x[k][j], temp1);
                x.array[k][j] = x.array[k][j] - temp1;
            }
        }
    }
    return true;
    
}/* doGauss */


static bool doReduce(matfp& x, INT32 ii, INT32 jj)
{
    fp             temp, temp1;
    INT32          i, j, k;
    bool            isZero;
    
    if(ii<0 || jj<0 || ii>=x.nr || jj>=x.nc)
        return false;
    
    k = 0; // to avoid incorrect warning
    // is pivot zero?
    if(x.array[ii][jj]==0)
    {
        // must swap rows
        // find non-zero element below pivot in pivot column
        isZero = true;
        for(i=ii+1;i<x.nr;++i)
        {
            if(x.array[i][jj]!=0)
            {
                isZero = false;
                k = i;
                break;
            }
        }
        
        if(isZero)
            return false;
        
        // have non-zero element in row k, so swap rows ii with k
        matfpSwapRows(x, ii, k);
    }
    // now we do the reduce operation
    // pivot element xt.array[ii][jj] is not zero
    // divide each element of row ii by xt.array[ii][jj] making xt.array[ii][jj] = 1
    //temp.Re = x[ii][jj].Re;
    //temp.Im = x[ii][jj].Im;
    temp = x.array[ii][jj];
    for(j=0;j<x.nc;++j)
    {
        //DivCC(x[ii][j], x[ii][j], temp);
        x.array[ii][j] = x.array[ii][j] / temp;
    }
    
    // now for every k not ii subtract row ii multiplied by xt[k][jj] from row k provided x[k][jj] != 0
    for(k=0;k<x.nr;++k)
    {
        if(k!=ii && x.array[k][jj]!=0)
        {
            //temp.Re = x[k][jj].Re;
            //temp.Im = x[k][jj].Im;
            temp = x.array[k][jj];
            for(j=0;j<x.nc;++j)
            {
                //MulCC(temp1, x[ii][j], temp);
                //SubCC(x[k][j], x[k][j], temp1);
                temp1 = x.array[ii][j] * temp;
                x.array[k][j] = x.array[k][j] - temp1;
            }
        }
    }
    return true;
    
}/* doReduce */

void myFree(matfp& x)
{
    INT32       i, j;
    
    if(x.array)
    {
        for(i=0;i<x.nr;++i)
        {
            for(j=0;j<x.nc;++j)
            {
                myFree(x.array[i][j]);
            }
            free(x.array[i]);
            x.array[i] = NULL;
        }
        free(x.array);
        x.array = NULL;
    }
    
}/* myFree */

// returns 1 if x=y else returns 0
INT32 matfpCompare(const matfp& x, const matfp& y)
{
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return 0;
    
    for(i=0;i<x.nr;i++)
        for(j=0;j<x.nc;j++)
            if(x.array[i][j]!=y.array[i][j])
                return 0;
    
    return 1;
    
}/* matfpCompare */



void add(matfp& z, const matfp& x, const matfp& y)
{
    matfp		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] + y.array[i][j];
    
    z = zt;
    
}/* add */


void sub(matfp& z, const matfp& x, const matfp& y)
{
    matfp		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] - y.array[i][j];
    
    z = zt;
    
}/* sub */


void mul(matfp& z, const matfp& x, const matfp& y)
{
    matfp		zt;
    INT32		i, j, k;
    
    if(x.nc!=y.nr)
        return;
    
    init(zt, x.nr, y.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
        {
            zt.array[i][j] = 0;
            for(k=0;k<x.nc;k++)
                zt.array[i][j] += x.array[i][k] * y.array[k][j];
        }
    
    z = zt;
    
}/* mul */


void mul(matfp& z, const matfp& x, const fp& y)
{
    matfp		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = y * x.array[i][j];
    
    z = zt;
    
}/* mul */

void div(matfp& z, const matfp& x, const matfp& y)
{
    matfp   yInv;
    fp      det;
    
    inv(yInv, det, y);
    if(!det)
        return;
    
    z = x * yInv;
    
}/* div */

void div(matfp& z, const matfp& x, const fp& y)
{
    matfp		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] / y;
    
    z = zt;
    
}/* div */

void mul(matfp& z, const fp& x, const matfp& y)
{
    mul(z, y, x);
    
}/* mul */

void mul(matfp& z, const matfp& x, const mb&  y)
{
    matfp		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = y * x.array[i][j];
    
    z = zt;
    
}/* mul */

void div(matfp& z, const matfp& x, const mb& y)
{
    matfp		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] / y;
    
    z = zt;
    
}/* div */

void mul(matfp& z, const mb& x, const matfp& y)
{
    mul(z, y, x);
    
}/* mul */


void mul(matfp& z, const matfp& x, double y)
{
    matfp		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = y * x.array[i][j];
    
    z = zt;
    
}/* mul */

void div(matfp& z, const matfp& x, double y)
{
    matfp		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] / y;
    
    z = zt;
    
}/* div */

void mul(matfp& z, double x, const matfp& y)
{
    mul(z, y, x);
    
}/* mul */

// creates unit matrix of size n
void matfpUnit(matfp& z, INT32 n)
{
    INT32		i, j;
    
    init(z, n, n);
    
    for(i=0;i<z.nr;i++)
        for(j=0;j<z.nc;j++)
        {
            if(i==j)
                z.array[i][j] = 1.;
            else
                z.array[i][j] = 0.;
        }
    
}/* matfpUnit */

matfp matfpUnit(INT32 n)
{
    matfp		z;
    
    matfpUnit(z, n);
    
    return z;
    
}/* matfpUnit */

void transpose(matfp& z, const matfp& x)
{
    matfp		zt;
    INT32		i, j;
    
    init(zt, x.nc, x.nr);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[j][i];
    
    z = zt;
    
}/* transpose */


matfp transpose(const matfp& x)
{
    matfp		z;
    
    transpose(z, x);
    
    return z;
    
}/* transpose */

// x must be square
bool trace(fp& z, const matfp& x)
{
    fp			zt;
    INT32		i;
    
    if(x.nr != x.nc)
    {
        z = 0;
        return false;
    }
    
    zt = 0;
    for(i=0;i<x.nr;i++)
        zt+=x.array[i][i];
    
    z = zt;
    
    return true;
    
}/* trace */

fp trace(const matfp& x)
{
    fp		z;
    
    if(trace(z, x))
    {
        return z;
    }
    else
    {
        z = 0;
        return z;
    }
    
}/* trace */

// puts z.array in echelon form
void matfpGaussElim(matfp& z, INT32& swapParity, const matfp& x)
{
    matfp       xt;
    INT32       ii, jj;
    
    swapParity = 1;
    
    init(xt, x.nr, x.nc);
    
    xt = x;
    
    ii = jj = 0;
    while(ii<xt.nr && jj<xt.nc)
    {
        if(doGauss(xt, swapParity, ii, jj))
        {
            ii++;
            jj++;
        }
        else
            jj++;
    }
    
    z = xt;
    
}/* matfpGaussElim */


// reduced echelon form; x must be in echolon form
void matfpReduce(matfp& z, const matfp& x)
{
    matfp           xt;
    INT32           ii, jj;
    
    init(xt, x.nr, x.nc);
    
    xt = x;
    
    ii = jj = 0;
    while(ii<xt.nr && jj<xt.nc)
    {
        if(doReduce(xt, ii, jj))
        {
            ii++;
            jj++;
        }
        else
            jj++;
    }
    z = xt;
    
}/* matfpReduce */

// z = inverse(x)
void inv(matfp& z, fp& determinant, const matfp& x)
{
    matfp   xt;
    INT32	n, i, j, k, L, *ik, *jk;
    fp      amax, save, zero=0., one = 1.;
    
    if(x.nr!=x.nc)
    {
        determinant = 0;
        return;
    }
    
    
    n = x.nr;
    
    init(xt, x.nr, x.nc);
    xt = x;
    
    ik = (INT32 *)malloc(n*sizeof(INT32));
    jk = (INT32 *)malloc(n*sizeof(INT32));
    
    determinant = one;
    
    for(k=0; k<n; k++)
    {
        /* find largest element in abs value in rest of matrix */
        amax = zero;
        
        for(i=k; i<n; i++)
        {
            for(j=k; j<n; j++)
            {
                if(abs(xt.array[i][j]) > abs(amax))
                {
                    amax = xt.array[i][j];
                    ik[k] = i;
                    jk[k] = j;
                }
            }
        }
        
        if(abs(amax) == zero)
        {
            z.nr = z.nc = 0;
            z.array = NULL;
            determinant = 0;
            return;
        }
        
        /* interchange rows and columns to put amax in array(k,k) */
        i = ik[k];
        if(i>k)
        {
            for(j=0; j<n; j++)
            {
                save = xt.array[k][j];
                xt.array[k][j] = xt.array[i][j];
                xt.array[i][j] = -save;
            }
        }
        
        j = jk[k];
        if(j>k)
        {
            for(i=0; i<n; i++)
            {
                save = xt.array[i][k];
                xt.array[i][k] = xt.array[i][j];
                xt.array[i][j] = -save;
            }
        }
        
        /* accumulate elements of inverse matrix */
        for(i=0; i<n; i++)
        {
            if(i!=k)
                xt.array[i][k] = -xt.array[i][k]/amax;
        }
        
        for(i=0; i<n; i++)
        {
            for(j=0; j<n; j++)
            {
                if(i!=k && j!=k)
                    xt.array[i][j] = xt.array[i][j] + xt.array[i][k]*xt.array[k][j];
            }
        }
        
        for(j=0; j<n; j++)
        {
            if(j!=k)
                xt.array[k][j] = xt.array[k][j]/amax;
        }
        
        xt.array[k][k] = one/amax;
        determinant = determinant*amax;
    }
    
    /* restore ordering of the matrix */
    for(L=0; L<n; L++)
    {
        k = n - L - 1;
        j = ik[k];
        
        if(j>k)
        {
            for(i=0; i<n; i++)
            {
                save = xt.array[i][k];
                xt.array[i][k] = -xt.array[i][j];
                xt.array[i][j] = save;
            }
        }
        
        i = jk[k];
        
        if(i>k)
        {
            for(j=0; j<n; j++)
            {
                save = xt.array[k][j];
                xt.array[k][j] = -xt.array[i][j];
                xt.array[i][j] = save;
            }
        }
    }
    
    z = xt;
    
    free(ik);
    free(jk);
    
} /* inv */

matfp inv(const matfp& x)
{
    fp      det;
    matfp   z;
    inv(z, det, x);
    
    return z;
    
}/* inv */

fp determinant(const matfp& x)
{
    matfp   z;
    fp      det;
    
    inv(z, det, x);
    
    return det;
    
}/* determinant */

// z = x^y ; x must be a square matrix and y>=0
bool power(matfp& z, const matfp& x, INT32 y)
{
    INT32				i, n, sy;
    static UINT32		mask; // 0x80000000
    //static mb			one;
    matfp				zt;
    static bool			initGood=false;
    
    if(!initGood)
    {
        //init(one, 1);
        mask = 1;
        mask = (mask<<31); // 0x80000000
        initGood = true;
    }
    
    if(x.nr!=x.nc || y<0)
    {
        //z = 0;
        return false;
    }
    
    if(matfpCompare(x, matfpUnit(x.nr)))
    {
        z = matfpUnit(x.nr);
        return true;
    }
    
    if(y==0)
    {
        z = matfpUnit(x.nr);
        return true;
    }
    
    zt = matfpUnit(x.nr);
    sy = y;
    n = NumBits(sy);
    
    sy = (sy<<(32-n));  // go to leading 1 bit
    
    for(i=0;i<n;i++)
    {
        mul(zt, zt, zt); // zt^2
        if(sy & mask)
            mul(zt, zt, x);
        sy = (sy<<1);
    }
    
    z = zt;
    
    return true;
    
}/* power */

// puts matrix y.array at bottom of x.array; the matrices must have the same number of columns
void matfpRowAugment(matfp& z, const matfp& x, const matfp& y)
{
    matfp		zt;
    INT32		i, j;
    
    if(x.nc!=y.nc)
        return;
    
    init(zt, x.nr+y.nr, x.nc);
    
    // copy x.array into zt.array top
    for(i=0;i<x.nr;i++)
        for(j=0;j<x.nc;j++)
            zt.array[i][j] = x.array[i][j];
    
    // copy y.array to bottom of zt.array
    for(i=0;i<y.nr;i++)
        for(j=0;j<y.nc;j++)
            zt.array[i+x.nr][j] = y.array[i][j];
    
    z = zt;
    
}/* matfpRowAugment */


matfp matfpRowAugment(const matfp& x, const matfp& y)
{
    matfp	z;
    
    matfpRowAugment(z, x, y);
    
    return z;
    
}/* matfpRowAugment */


// puts matrix y.array to right of x.array; the matrices must have the same number of rows
void matfpColAugment(matfp& z, const matfp& x, const matfp& y)
{
    matfp		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr)
        return;
    
    init(zt, x.nr, x.nc+y.nc);
    
    zt.nr = x.nr;
    zt.nc = x.nc + y.nc;
    
    // copy x.array into zt.array left
    for(i=0;i<x.nr;i++)
        for(j=0;j<x.nc;j++)
            zt.array[i][j] = x.array[i][j];
    
    // copy y.array to right of zt.array
    for(i=0;i<y.nr;i++)
        for(j=0;j<y.nc;j++)
            zt.array[i][j+x.nc] = y.array[i][j];
    
    z = zt;
    
}/* matfpColAugment */

matfp matfpColAugment(const matfp& x, const matfp& y)
{
    matfp	z;
    
    matfpColAugment(z, x, y);
    
    return z;
    
}/* matfpColAugment */

// creates permutation matrix which, x from left, interchanges row1 with row2
static matfp createPermutationMatrix(INT32 n, INT32 row1, INT32 row2)
{
    matfp    P;
    
    if (n<1)
    {
        n = 1;
    }
    
    P = matfpUnit(n);
    if (row1==row2 || row1<0 || row2<0 || row1>=n || row2>=n)
    {
        return P;
    }
    
    P.array[row1][row1] = 0;
    P.array[row2][row2] = 0;
    P.array[row1][row2] = 1;
    P.array[row2][row1] = 1;
    return P;
    
}/* createPermutationMatrix */

// adds transpose permutation Q to P, returns Q
static matfp addTranspose(matfp& P, INT32 row1,  INT32 row2, INT32& swapParity)
{
    INT32   n;
    matfp    Q;
    
    n = P.nc;
    if (row1==row2 || row1<0 || row2<0 || row1>=n || row2>=n)
    {
        Q = matfpUnit(n);
        return Q;
    }
    
    Q = createPermutationMatrix(n, row1, row2);
    
    P = Q * P;
    swapParity = -swapParity;
    
    return Q;
    
}/* addTranspose */


// if A has zeroes on diagonal, PA does not
static bool permutationMatrix(const matfp& A, matfp& P, INT32& swapParity)
{
    INT32   n, i, j, k;
    matfp    B, Q;
    bool    isZero;
    fp      temp;
    INT32   count, countMax=362880; // 10! / 10
    
    swapParity = 1;
    
    n = A.nc;
    P = matfpUnit(n);
    
    if (n!=A.nr)
    {
        return false;
    }
    
    isZero = false;
    for (i=0; i<n; ++i)
    {
        if (A.array[i][i]==0)
        {
            isZero = true;
            break;
        }
    }
    if (!isZero)
    {
        return true; // P is unit matrix
    }
    
    
    B = A;
    isZero = true;
    count = 0;
    while(isZero && count<countMax)
    {
        count++;
        
        for (j=0; j<n; ++j)
        {
            if (B.array[j][j]==0)
            {
                temp = 0;
                for (i=0; i<n; ++i)
                {
                    if (abs(B.array[i][j])>temp)
                    {
                        temp = abs(B.array[i][j]);
                        k = i;
                    }
                }
                if (temp==0)
                {
                    return false;
                }
                Q = addTranspose(P, j, k, swapParity);  // k >is< initialized
                B = Q * B;
            }
        }
        // some A's will still give a B with a zero on diagonal
        isZero = false;
        for (i=0; i<n; ++i)
        {
            if (B.array[i][i]==0)
            {
                isZero = true;
                break;
            }
        }
    }
    
    if(isZero)
        return false;
    
    return true;
    
}/* permutationMatrix */

// Doolittle algorithm, L is unit lower diagonal; A = P L U
bool LUdecomp(const matfp& A, matfp& P, matfp& L, matfp& U, INT32& swapParity)
{
    INT32   n, i, j, k;
    fp      alpha;
    
    if(!determinant(A))
        return false;
    
    if(!permutationMatrix(A, P, swapParity))
    {
        return false;
    }
    
    U = P * A;
    
    n = U.nr;
    for (i=0; i<n; ++i)
    {
        for (j=0; j<i; ++j)
        {
            alpha = U.array[i][j];
            for (k=0; k<j; ++k)
            {
                alpha = alpha - U.array[i][k] * U.array[k][j];
            }
            U.array[i][j] = alpha / U.array[j][j];
        }
        
        for (j=i; j<n; ++j)
        {
            alpha = U.array[i][j];
            for (k=0; k<i; ++k)
            {
                alpha = alpha - U.array[i][k] * U.array[k][j];
            }
            U.array[i][j] = alpha;
        }
    }
    
    // I -> L; modify U
    L = matfpUnit(n);
    for (j=0; j<n; ++j)
    {
        for (i=j+1; i<n; ++i)
        {
            L.array[i][j] = U.array[i][j];
            U.array[i][j] = 0;
        }
    }
    
    P = transpose(P);
    
    return true;
    
}/* LUdecomp */

static bool isRowZero(const matfp& x, long row, const fp& zeroSet)
{
    bool    isZero;
    long    j;
    
    isZero = true;
    for(j=0;j<x.nc;j++)
    {
        if(abs(x.array[row][j])>zeroSet)
        {
            isZero = false;
            break;
        }
    }
    return isZero;
    
}/* isRowZero */

static void zeroOut(matfp& z, const matfp& x, const fp& zeroSet)
{
    long    i, j;
    
    z = x;
    for(i=0;i<z.nr;++i)
        for(j=0;j<z.nc;++j)
            if(abs(z.array[i][j])<=zeroSet)
                z.array[i][j] = 0;
    
}/* zeroOut */

// effectiveZero is positive small valued double considered as zero
long matfpRank(const matfp& x, const fp& zeroSet)
{
    long    rank, i;
    matfp   z, zt, xt;
    int     swapParity;
    
    rank = 0;
    if(!x.nr)
        return rank;
    
    if(x.nr==1)
    {
        if(isRowZero(x, x.nr, zeroSet))
            return 0;
        else
            return 1;
    }
    matfpGaussElim(xt, swapParity, x);
    // need to zero out xt using effectiveZero
    zeroOut(zt, xt, zeroSet);
    matfpReduce(z, zt);
    
    for(i=0;i<z.nr;++i)
    {
        if(!isRowZero(z, i, zeroSet))
            rank++;
    }
    return rank;
    
}/* matfpRank */


void matfpEigenvectors(matc& eigenvectors, matc& eigenvalues, long*& algMultiplicity, long*& geoMultiplicity, const matfp& x, long myDecPrec, const fp& zeroSet)
{
    long        i, j;
    matc        xc;
    
    if(x.nr!=x.nc || x.nr<1)
        return;
    
    init(xc, x.nr, x.nc);
    
    for(i=0;i<x.nr;++i)
        for(j=0;j<x.nc;++j)
            xc.array[i][j] = x.array[i][j];
    
    matcEigenvectors(eigenvectors, eigenvalues, algMultiplicity, geoMultiplicity, xc, myDecPrec, zeroSet);
    
}/* matfpEigenvectors */
