//
//  matcMath.cpp
//  fpComplex
//
//  Created by Robert Delaney on 2/2/17.
//
//

#include "matcMath.hpp"
#include "matcConv.hpp"
#include "matfp.hpp"
#include "findPolyRoots.h"

// swap row i with row j
static void matcSwapRows(matc& x, INT32 i, INT32 j)
{
    fpComplex 		temp;
    INT32           k;
    
    for(k=0;k<x.nc;k++)
    {
        temp = x.array[i][k];
        x.array[i][k] = x.array[j][k];
        x.array[j][k] = temp;
    }
    
}/* matcSwapRows */

// if an abs(fp)<zeroSet, fp=0
static bool doGauss(matc& x, INT32& swapParity, INT32 ii, INT32 jj, const fp& zeroSet)
{
    fpComplex   temp, temp1;
    INT32       i, j, k;
    bool        isZero;
    
    if(ii<0 || jj<0 || ii>=x.nr || jj>=x.nc)
        return false;
    
    k = 0;
    
    // is pivot zero?
    if(!(abs(x.array[ii][jj].re)<zeroSet && abs(x.array[ii][jj].im)<zeroSet))
    {
        // must swap rows
        // find non-zero element below pivot in pivot column
        isZero = true;
        for(i=ii+1;i<x.nr;++i)
        {
            if(abs(x.array[i][jj].re)>zeroSet || abs(x.array[i][jj].im)>zeroSet)
            {
                isZero = false;
                k = i;
                break;
            }
        }
        
        if(isZero)
            return false;
        
        // have non-zero element in row k, so swap rows ii with k
        matcSwapRows(x, ii, k);
        swapParity = -swapParity;
    }
    else
    {
        x.array[ii][jj].re = 0;
        x.array[ii][jj].im = 0;
    }
    
    // now we do the Gauss Elim. operation
    // pivot element xt[ii][jj] is not zero
    
    // now for every k not ii subtract row ii multiplied by xt[k][jj] from row k provided x[k][jj] != 0
    for(k=0;k<x.nr;++k)
    {
        if(k!=ii && (x.array[k][jj].re!=0 || x.array[k][jj].im!=0))
        {
            //temp = x[k][jj] / x[ii][jj];
            //DivCC(temp, x[k][jj], x[ii][jj]);
            temp = x.array[k][jj] / x.array[ii][jj];
            for(j=0;j<x.nc;++j)
            {
                //MulCC(temp1, x[ii][j], temp);
                temp1 = x.array[ii][j] * temp;
                //SubCC(x[k][j], x[k][j], temp1);
                x.array[k][j] = x.array[k][j] - temp1;
            }
        }
    }
    return true;
    
}/* doGauss */

// if an abs(fp)<zeroSet, fp=0
static bool doReduce(matc& x, INT32 ii, INT32 jj, const fp& zeroSet)
{
    fpComplex      temp, temp1;
    INT32          i, j, k;
    bool           isZero;
    
    if(ii<0 || jj<0 || ii>=x.nr || jj>=x.nc)
        return false;
    
    k = 0; // to avoid incorrect warning
    // is pivot zero?
    if(abs(x.array[ii][jj].re)<zeroSet && abs(x.array[ii][jj].im)<zeroSet)
    {
        x.array[ii][jj].re = 0;
        x.array[ii][jj].im = 0;
        // must swap rows
        // find non-zero element below pivot in pivot column
        isZero = true;
        for(i=ii+1;i<x.nr;++i)
        {
            if(abs(x.array[i][jj].re)>zeroSet || abs(x.array[i][jj].im)>zeroSet)
            {
                isZero = false;
                k = i;
                break;
            }
        }
        
        if(isZero)
            return false;
        
        // have non-zero element in row k, so swap rows ii with k
        matcSwapRows(x, ii, k);
    }
    // now we do the reduce operation
    // pivot element xt.array[ii][jj] is not zero
    // divide each element of row ii by xt.array[ii][jj] making xt.array[ii][jj] = 1
    //temp.Re = x[ii][jj].Re;
    //temp.Im = x[ii][jj].Im;
    temp = x.array[ii][jj];
    for(j=0;j<x.nc;++j)
    {
        //DivCC(x[ii][j], x[ii][j], temp);
        x.array[ii][j] = x.array[ii][j] / temp;
    }
    
    // now for every k not ii subtract row ii multiplied by xt[k][jj] from row k provided x[k][jj] != 0
    for(k=0;k<x.nr;++k)
    {
        if(k!=ii && (abs(x.array[k][jj].re)>zeroSet || abs(x.array[k][jj].im)>zeroSet))
        {
            //temp.Re = x[k][jj].Re;
            //temp.Im = x[k][jj].Im;
            temp = x.array[k][jj];
            for(j=0;j<x.nc;++j)
            {
                //MulCC(temp1, x[ii][j], temp);
                //SubCC(x[k][j], x[k][j], temp1);
                temp1 = x.array[ii][j] * temp;
                x.array[k][j] = x.array[k][j] - temp1;
            }
        }
    }
    return true;
    
}/* doReduce */

void myFree(matc& x)
{
    INT32       i, j;
    
    if(x.array)
    {
        for(i=0;i<x.nr;++i)
        {
            for(j=0;j<x.nc;++j)
            {
                myFree(x.array[i][j]);
            }
            free(x.array[i]);
            x.array[i] = NULL;
        }
        free(x.array);
        x.array = NULL;
    }
    
}/* myFree */

// returns 1 if x=y else returns 0
INT32 matcCompare(const matc& x, const matc& y)
{
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return 0;
    
    for(i=0;i<x.nr;i++)
        for(j=0;j<x.nc;j++)
            if(x.array[i][j]!=y.array[i][j])
                return 0;
    
    return 1;
    
}/* matcCompare */


void add(matc& z, const matc& x, const matc& y)
{
    matc		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] + y.array[i][j];
    
    z = zt;
    
}/* add */

void add(matc& z, const matc& x, const matfp& y)
{
    matc		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] + y.array[i][j];
    
    z = zt;
    
}/* add */

void add(matc& z, const matfp& x, const matc& y)
{
    add(z, y, x);
    
}/* add */


void sub(matc& z, const matc& x, const matc& y)
{
    matc		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] - y.array[i][j];
    
    z = zt;
    
}/* sub */

void sub(matc& z, const matc& x, const matfp& y)
{
    matc		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] - y.array[i][j];
    
    z = zt;
    
}/* sub */

void sub(matc& z, const matfp& x, const matc& y)
{
    matc		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr || x.nc!=y.nc)
        return;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] - y.array[i][j];
    
    z = zt;
    
}/* sub */


void mul(matc& z, const matc& x, const matc& y)
{
    matc		zt;
    INT32		i, j, k;
    
    if(x.nc!=y.nr)
        return;
    
    init(zt, x.nr, y.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
        {
            zt.array[i][j].re = 0;
            zt.array[i][j].im = 0;
            for(k=0;k<x.nc;k++)
                zt.array[i][j] += x.array[i][k] * y.array[k][j];
        }
    
    z = zt;
    
}/* mul */

void mul(matc& z, const matc& x, const matfp& y)
{
    matc		zt;
    INT32		i, j, k;
    
    if(x.nc!=y.nr)
        return;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
        {
            zt.array[i][j].re = 0;
            zt.array[i][j].im = 0;
            for(k=0;k<x.nc;k++)
                zt.array[i][j] += x.array[i][k] * y.array[k][j];
        }
    
    z = zt;
    
}/* mul */

void mul(matc& z, const matfp& x, const matc& y)
{
    mul(z, y, x);
    
}/* mul */


void mul(matc& z, const matc& x, const fp& y)
{
    matc		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = y * x.array[i][j];
    
    z = zt;
    
}/* mul */

void mul(matc& z, const matc& x, const fpComplex& y)
{
    matc        zt;
    INT32        i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = y * x.array[i][j];
    
    z = zt;
    
}/* mul */

void div(matc& z, const matc& x, const fp& y)
{
    matc		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] / y;
    
    z = zt;
    
}/* div */

void mul(matc& z, const fp& x, const matc& y)
{
    mul(z, y, x);
    
}/* mul */

void mul(matc& z, const fpComplex& x, const matc& y)
{
    mul(z, y, x);
    
}/* mul */

void mul(matc& z, const matc& x, const mb&  y)
{
    matc		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = y * x.array[i][j];
    
    z = zt;
    
}/* mul */

void div(matc& z, const matc& x, const mb& y)
{
    matc		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] / y;
    
    z = zt;
    
}/* div */

void mul(matc& z, const mb& x, const matc& y)
{
    mul(z, y, x);
    
}/* mul */


void mul(matc& z, const matc& x, double y)
{
    matc		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = y * x.array[i][j];
    
    z = zt;
    
}/* mul */

void div(matc& z, const matc& x, double y)
{
    matc		zt;
    INT32		i, j;
    
    init(zt, x.nr, x.nc);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[i][j] / y;
    
    z = zt;
    
}/* div */

void mul(matc& z, double x, const matc& y)
{
    mul(z, y, x);
    
}/* mul */

// creates unit matrix of size n
void matcUnit(matc& z, INT32 n)
{
    INT32		i, j;
    
    init(z, n, n);
    
    for(i=0;i<z.nr;i++)
        for(j=0;j<z.nc;j++)
        {
            if(i==j)
                z.array[i][j] = 1.;
            else
                z.array[i][j] = 0.;
        }
    
}/* matcUnit */

matc matcUnit(INT32 n)
{
    matc		z;
    
    matcUnit(z, n);
    
    return z;
    
}/* matcUnit */

void transpose(matc& z, const matc& x)
{
    matc		zt;
    INT32		i, j;
    
    init(zt, x.nc, x.nr);
    
    for(i=0;i<zt.nr;i++)
        for(j=0;j<zt.nc;j++)
            zt.array[i][j] = x.array[j][i];
    
    z = zt;
    
}/* transpose */


matc transpose(const matc& x)
{
    matc		z;
    
    transpose(z, x);
    
    return z;
    
}/* transpose */

// x must be square
bool trace(fpComplex& z, const matc& x)
{
    fpComplex	zt;
    INT32		i;
    
    if(x.nr != x.nc)
    {
        z.re = 0.;
        z.im = 0.;
        return false;
    }
    
    zt.re = 0.;
    zt.im = 0.;
    
    for(i=0;i<x.nr;i++)
        zt+=x.array[i][i];
    
    z = zt;
    
    return true;
    
}/* trace */


fpComplex trace(const matc& x)
{
    fpComplex		z;
    
    if(trace(z, x))
    {
        return z;
    }
    else
    {
        z.re = 0.;
        z.im = 0.;
        return z;
    }
    
}/* trace */


// puts z.array in echelon form
// if abs(fp)<zeroSet, fp=0
void matcGaussElim(matc& z, INT32& swapParity, const matc& x, const fp& zeroSet)
{
    matc        xt;
    INT32       ii, jj;
    
    swapParity = 1;
    
    init(xt, x.nr, x.nc);
    
    xt = x;
    
    ii = jj = 0;
    while(ii<xt.nr && jj<xt.nc)
    {
        if(doGauss(xt, swapParity, ii, jj, zeroSet))
        {
            ii++;
            jj++;
        }
        else
            jj++;
    }
    
    z = xt;
    
}/* matcGaussElim */


// reduced echelon form; x must be in echolon form
// if abs(fp)<zeroSet, fp=0
void matcReduce(matc& z, const matc& x, const fp& zeroSet)
{
    matc            xt;
    INT32           ii, jj;
    
    init(xt, x.nr, x.nc);
    
    xt = x;
    
    ii = jj = 0;
    while(ii<xt.nr && jj<xt.nc)
    {
        if(doReduce(xt, ii, jj, zeroSet))
        {
            ii++;
            jj++;
        }
        else
            jj++;
    }
    z = xt;
    
}/* matcReduce */


// z = inverse(x)
void inv(matc& z, fpComplex& determinant, const matc& x)
{
    matc            xt;
    INT32           n, i, j, k, L, *ik, *jk;
    fpComplex       amax, save, zero=0., one = 1.;
    
    if(x.nr!=x.nc)
        return;
    
    n = x.nr;
    
    init(xt, x.nr, x.nc);
    xt = x;
    
    ik = (INT32 *)malloc(n*sizeof(INT32));
    jk = (INT32 *)malloc(n*sizeof(INT32));
    
    determinant = one;
    
    for(k=0; k<n; k++)
    {
        /* find largest element in abs value in rest of matrix */
        amax = zero;
        
        for(i=k; i<n; i++)
        {
            for(j=k; j<n; j++)
            {
                if(abs(xt.array[i][j]) > abs(amax))
                {
                    amax = xt.array[i][j];
                    ik[k] = i;
                    jk[k] = j;
                }
            }
        }
        
        if(!abs(amax))
        {
            z.nr = z.nc = 0;
            z.array = NULL;
            determinant.re = 0;
            determinant.im = 0;
            return;
        }
        
        /* interchange rows and columns to put amax in array(k,k) */
        i = ik[k];
        if(i>k)
        {
            for(j=0; j<n; j++)
            {
                save = xt.array[k][j];
                xt.array[k][j] = xt.array[i][j];
                xt.array[i][j] = -save;
            }
        }
        
        j = jk[k];
        if(j>k)
        {
            for(i=0; i<n; i++)
            {
                save = xt.array[i][k];
                xt.array[i][k] = xt.array[i][j];
                xt.array[i][j] = -save;
            }
        }
        
        /* accumulate elements of inverse matrix */
        for(i=0; i<n; i++)
        {
            if(i!=k)
                xt.array[i][k] = -xt.array[i][k]/amax;
        }
        
        for(i=0; i<n; i++)
        {
            for(j=0; j<n; j++)
            {
                if(i!=k && j!=k)
                    xt.array[i][j] = xt.array[i][j] + xt.array[i][k]*xt.array[k][j];
            }
        }
        
        for(j=0; j<n; j++)
        {
            if(j!=k)
                xt.array[k][j] = xt.array[k][j]/amax;
        }
        
        xt.array[k][k] = one/amax;
        determinant = determinant*amax;
    }
    
    /* restore ordering of the matrix */
    for(L=0; L<n; L++)
    {
        k = n - L - 1;
        j = ik[k];
        
        if(j>k)
        {
            for(i=0; i<n; i++)
            {
                save = xt.array[i][k];
                xt.array[i][k] = -xt.array[i][j];
                xt.array[i][j] = save;
            }
        }
        
        i = jk[k];
        
        if(i>k)
        {
            for(j=0; j<n; j++)
            {
                save = xt.array[k][j];
                xt.array[k][j] = -xt.array[i][j];
                xt.array[i][j] = save;
            }
        }
    }
    
    z = xt;
    
    free(ik);
    free(jk);
    
} /* inv */

matc inv(const matc& x)
{
    fpComplex      det;
    matc   z;
    inv(z, det, x);
    
    return z;
    
}/* inv */

fpComplex determinant(const matc& x)
{
    matc            z;
    fpComplex      det;
    
    inv(z, det, x);
    
    return det;
    
}/* determinant */


// z = x^y ; x must be a square matrix and y>=0
bool power(matc& z, const matc& x, INT32 y)
{
    INT32				i, n, sy;
    static UINT32		mask; // 0x80000000
    //static mb			one;
    matc				zt;
    static bool			initGood=false;
    
    if(!initGood)
    {
        //init(one, 1);
        mask = 1;
        mask = (mask<<31); // 0x80000000
        initGood = true;
    }
    
    if(x.nr!=x.nc || y<0)
    {
        //z = 0;
        return false;
    }
    
    if(matcCompare(x, matcUnit(x.nr)))
    {
        z = matcUnit(x.nr);
        return true;
    }
    
    if(y==0)
    {
        z = matcUnit(x.nr);
        return true;
    }
    
    zt = matcUnit(x.nr);
    sy = y;
    n = NumBits(sy);
    
    sy = (sy<<(32-n));  // go to leading 1 bit
    
    for(i=0;i<n;i++)
    {
        mul(zt, zt, zt); // zt^2
        if(sy & mask)
            mul(zt, zt, x);
        sy = (sy<<1);
    }
    
    z = zt;
    
    return true;
    
}/* power */

// puts matrix y.array at bottom of x.array; the matrices must have the same number of columns
void matcRowAugment(matc& z, const matc& x, const matc& y)
{
    matc		zt;
    INT32		i, j;
    
    if(x.nc!=y.nc)
        return;
    
    init(zt, x.nr+y.nr, x.nc);
    
    // copy x.array into zt.array top
    for(i=0;i<x.nr;i++)
        for(j=0;j<x.nc;j++)
            zt.array[i][j] = x.array[i][j];
    
    // copy y.array to bottom of zt.array
    for(i=0;i<y.nr;i++)
        for(j=0;j<y.nc;j++)
            zt.array[i+x.nr][j] = y.array[i][j];
    
    z = zt;
    
}/* matcRowAugment */


matc matfpRowAugment(const matc& x, const matc& y)
{
    matc	z;
    
    matcRowAugment(z, x, y);
    
    return z;
    
}/* matcRowAugment */


// puts matrix y.array to right of x.array; the matrices must have the same number of rows
void matcColAugment(matc& z, const matc& x, const matc& y)
{
    matc		zt;
    INT32		i, j;
    
    if(x.nr!=y.nr)
        return;
    
    init(zt, x.nr, x.nc+y.nc);
    
    zt.nr = x.nr;
    zt.nc = x.nc + y.nc;
    
    // copy x.array into zt.array left
    for(i=0;i<x.nr;i++)
        for(j=0;j<x.nc;j++)
            zt.array[i][j] = x.array[i][j];
    
    // copy y.array to right of zt.array
    for(i=0;i<y.nr;i++)
        for(j=0;j<y.nc;j++)
            zt.array[i][j+x.nc] = y.array[i][j];
    
    z = zt;
    
}/* matfpColAugment */

matc matcColAugment(const matc& x, const matc& y)
{
    matc	z;
    
    matcColAugment(z, x, y);
    
    return z;
    
}/* matcColAugment */

// creates permutation matrix which, x from left, interchanges row1 with row2
static matc createPermutationMatrix(INT32 n, INT32 row1, INT32 row2)
{
    matc    P;
    
    if (n<1)
    {
        n = 1;
    }
    
    P = matcUnit(n);
    if (row1==row2 || row1<0 || row2<0 || row1>=n || row2>=n)
    {
        return P;
    }
    
    P.array[row1][row1] = 0.;
    P.array[row2][row2] = 0.;
    P.array[row1][row2] = 1;
    P.array[row2][row1] = 1;
    return P;
    
}/* createPermutationMatrix */

// adds transpose permutation Q to P, returns Q
static matc addTranspose(matc& P, INT32 row1,  INT32 row2, INT32& swapParity)
{
    INT32   n;
    matc    Q;
    
    n = P.nc;
    if (row1==row2 || row1<0 || row2<0 || row1>=n || row2>=n)
    {
        Q = matcUnit(n);
        return Q;
    }
    
    Q = createPermutationMatrix(n, row1, row2);
    
    P = Q * P;
    swapParity = -swapParity;
    
    return Q;
    
}/* addTranspose */


// if A has zeroes on diagonal, PA does not
static bool permutationMatrix(const matc& A, matc& P, INT32& swapParity)
{
    INT32           n, i, j, k;
    matc            B, Q;
    bool            isZero;
    fp              temp;
    INT32           count, countMax=362880; // 10! / 10
    
    swapParity = 1;
    
    n = A.nc;
    P = matcUnit(n);
    
    if (n!=A.nr)
    {
        return false;
    }
    
    isZero = false;
    for (i=0; i<n; ++i)
    {
        if (A.array[i][i]==0)
        {
            isZero = true;
            break;
        }
    }
    if (!isZero)
    {
        return true; // P is unit matrix
    }
    
    
    B = A;
    isZero = true;
    count = 0;
    while(isZero && count<countMax)
    {
        count++;
        
        for (j=0; j<n; ++j)
        {
            if (B.array[j][j]==0)
            {
                temp = 0.;
                for (i=0; i<n; ++i)
                {
                    if (abs(B.array[i][j])>temp)
                    {
                        temp = abs(B.array[i][j]);
                        k = i;
                    }
                }
                if (temp==0)
                {
                    return false;
                }
                Q = addTranspose(P, j, k, swapParity);  // k >is< initialized
                B = Q * B;
            }
        }
        // some A's will still give a B with a zero on diagonal
        isZero = false;
        for (i=0; i<n; ++i)
        {
            if (B.array[i][i]==0)
            {
                isZero = true;
                break;
            }
        }
    }
    
    
    if(isZero)
        return false;
    
    return true;
    
}/* permutationMatrix */

// Doolittle algorithm, L is unit lower diagonal; A = P L U
bool LUdecomp(const matc& A, matc& P, matc& L, matc& U, INT32& swapParity)
{
    INT32   n, i, j, k;
    fpComplex      alpha;
    
    if(!determinant(A))
        return false;
    
    if(!permutationMatrix(A, P, swapParity))
    {
        return false;
    }
    
    U = P * A;
    
    n = U.nr;
    for (i=0; i<n; ++i)
    {
        for (j=0; j<i; ++j)
        {
            alpha = U.array[i][j];
            for (k=0; k<j; ++k)
            {
                alpha = alpha - U.array[i][k] * U.array[k][j];
            }
            U.array[i][j] = alpha / U.array[j][j];
        }
        
        for (j=i; j<n; ++j)
        {
            alpha = U.array[i][j];
            for (k=0; k<i; ++k)
            {
                alpha = alpha - U.array[i][k] * U.array[k][j];
            }
            U.array[i][j] = alpha;
        }
    }
    
    // I -> L; modify U
    L = matcUnit(n);
    for (j=0; j<n; ++j)
    {
        for (i=j+1; i<n; ++i)
        {
            L.array[i][j] = U.array[i][j];
            U.array[i][j] = 0.;
        }
    }
    
    P = transpose(P);
    
    return true;
    
}/* LUdecomp */

static bool isRowZero(const matc& x, long row, const fp& zeroSet)
{
    bool    isZero;
    long    j;
    
    isZero = true;
    for(j=0;j<x.nc;j++)
    {
        if(abs(x.array[row][j])>zeroSet)
        {
            isZero = false;
            break;
        }
    }
    return isZero;
    
}/* isRowZero */

static void zeroOut(matc& z, const matc& x, const fp& zeroSet)
{
    long    i, j;
    
    z = x;
    for(i=0;i<z.nr;++i)
        for(j=0;j<z.nc;++j)
            if(abs(z.array[i][j])<=zeroSet)
            {
                z.array[i][j].re = 0;
                z.array[i][j].im = 0;
            }

}/* zeroOut */

// zeroSet is positive small valued fp considered as zero
long matcRank(const matc& x, const fp& zeroSet)
{
    long    rank, i;
    matc    z, zt, xt;
    int     swapParity;
    
    rank = 0;
    if(!x.nr)
        return rank;
    
    if(x.nr==1)
    {
        if(isRowZero(x, x.nr, zeroSet))
            return 0;
        else
            return 1;
    }
    matcGaussElim(xt, swapParity, x, zeroSet);
    // need to zero out xt using zeroSet
    zeroOut(zt, xt, zeroSet);
    matcReduce(z, zt, zeroSet);
    
    for(i=0;i<z.nr;++i)
    {
        if(!isRowZero(z, i, zeroSet))
            rank++;
    }
    return rank;
    
}/* matcRank */

/*
 Finds the characteristic polynomial of a square matrix using a method described by
 Prof. Bumby of Rutgers Univ.
 
 Let M be the matrix of size nxn, and the characteristic polynomial be:
 
 x^n - p[1]*x^n-1 - p[2]*x^n-2 - ... - p[n], and let p[0] = 1
 
 Then define matrices F such that
 
 F_1(M) = M
 
 and for k = 1, 2,...,n
 
 p[k] = (1/k)*Tr(F_k)
 
 F_k+1 = M*(F_k - p[k]*I)
 */
bool charPoly(fpComplex*& z, const matc& x)
{
    matc            F, I;
    int             k, m; // size of x matrix and fpComplex array
    int             i;
    
    if(x.nr!=x.nc || !x.nr)
        return false;
    
    m = x.nr + 1; // degree of char. poly
    z = (fpComplex*)malloc(m*sizeof(fpComplex));
    for(i=0;i<m;++i)
        init(z[i]);
    
    z[m-1] = 1;
    
    I = matcUnit(x.nr);
    
    // -p[k] <-> zt[m-k]
    F = x;
    for(k=1;k<m;++k)
    {
        z[m-k-1] = -trace(F) / k;
        F = x*(F + I*z[m-k-1]);
    }

    return true;
    
}/* charPoly */

// z will hold the eigenvalues of x
// pArray holds poly(root) for each root
// roots will always be found with blockPrec=16 unless myDecPrec makes it bigger
bool matcEigenvalues(fpComplex*& z, fpComplex*& pArray, const matc& x, long myDecPrec)
{
    fpComplex   *p;  // holds coeffs. of characteristic poly of x
    
    long        n;
    int         saveBlockPrec;
    
    if(x.nr!=x.nc || x.nr<1)
        return false;
    
    saveBlockPrec = getBlockPrec();
    if(myDecPrec<129)
        setBlockPrec(16);
    else
        setBlockPrec(myDecPrec/8 + 1);
                     
    n = x.nr; // size of matrix
    
    charPoly(p, x);
    
    if(!findPolyRoots(z, pArray, p, n, true))
    {
        setBlockPrec(saveBlockPrec);
        return false;
    }
    
    setBlockPrec(saveBlockPrec);
    return true;
    
}/* matcEigenvalues */

// bool findPolyRoots(fpComplex*& root, fpComplex*& Parray, fpComplex*& coeffArray, long n, bool doRoundToZero)

// https://en.wikipedia.org/wiki/Kernel_(linear_algebra)
void matcNullSpace(INT32& numNV, matc& nullVectors, const matc& x, const fp& zeroSet)
{
    matc        xt, yt, zt;
    matc        reduced;
    INT32        i, j, k;
    INT32        swapParity;
    
    xt = x;
    
    matcRowAugment(yt, xt, matcUnit(xt.nc));
    matcGaussElim(zt, swapParity, transpose(yt), zeroSet);
    matcReduce(reduced, zt, zeroSet);
    reduced = transpose(reduced); // x.nr+x.nc x x.nc
    // use zeroSet
    for(i=0;i<reduced.nr;++i)
        for(j=0;j<reduced.nc;++j)
        {
            if(abs(reduced.array[i][j].re)<zeroSet)
                reduced.array[i][j].re = 0;
            if(abs(reduced.array[i][j].im)<zeroSet)
                reduced.array[i][j].im = 0;
        }

    // look for zero columns in reduced for row index from 0 to xt.nr; null vectors are at the bottems of these columns and have xt.nc elements
    // create a Boolean array to hold true or false for null vector columns
    bool    *hasNV;
    bool    isNull;
    hasNV = (bool*)malloc(xt.nc*sizeof(bool));
    for(j=0;j<xt.nc;j++)
    {
        isNull = true;
        for(i=0;i<xt.nr;i++)
        {
            if(abs(reduced.array[i][j].re)>zeroSet || abs(reduced.array[i][j].im)>zeroSet)
            {
                isNull = false;
                break;
            }
        }
        if(isNull)
            hasNV[j] = true;
        else
            hasNV[j] = false;
    }
    
    // get number of null vectors
    numNV = 0;
    for(i=0;i<xt.nc;i++)
    {
        if(hasNV[i])
            numNV++;
    }
    
    if(!numNV)
    {
        free(hasNV);
        init(nullVectors, 1, 1);
        nullVectors.array[0][0].re = 0;
        nullVectors.array[0][0].im = 0;
        return;
    }
    
    // now have null vectors which we want as columns of myNullVectors.array
    nullVectors.nr = xt.nc;
    nullVectors.nc = numNV;  // column size of xt
    init(nullVectors, nullVectors.nr, nullVectors.nc);
    //cout << nullVectors.nc << endl;
    // copy null vectors into columns of nullVectors.array and get lcm of den's and multiply after that
    k = 0;
    for(j=0;j<xt.nc;j++)
    {
        if(hasNV[j])
        {
            for(i=0;i<nullVectors.nr;i++)
                nullVectors.array[i][k] = reduced.array[xt.nr+i][j];
            k++;
        }
    }
    free(hasNV);
    
}/* matcNullSpace */

// use eigenvalues eList and find algebraic multiplicities
// all matc's are column matrices as vectors
static void adjust(matc& eigenvalues, long*& multiplicity, const matc& eList, const fp& zeroSet)
{
    matc        z; // temp holders of eigenvalues
    long        i, j, js,*mult, numE;
    long        k; // pointer to z
    fpComplex   c, temp;
    bool        isDup;
    
    init(z, eList.nr, 1);
    mult = (long*)malloc(eList.nr*sizeof(long));
    
    z.array[0][0] = eList.array[0][0];
    k = 0;
    mult[0] = 1;
    numE = 1;
    for(i=1;i<eList.nr;++i)
    {
        c = eList.array[i][0];
        isDup = false;
        for(j=0;j<numE;++j)
        {
            temp = z.array[j][0];
            if(abs(c.re-temp.re)<zeroSet && abs(c.im-temp.im)<zeroSet)
            {
                js = j;
                isDup = true;
            }
        }
        if(!isDup)
        {
            k++;
            z.array[k][0] = c;
            mult[k] = 1;
            numE++;
        }
        else
            mult[js]++;
    }
    init(eigenvalues, numE, 1);
    multiplicity = (long*)malloc(numE*sizeof(long));
    for(i=0;i<numE;++i)
    {
        eigenvalues.array[i][0] = z.array[i][0];
        multiplicity[i] = mult[i];
    }
    myFree(z);
    free(mult);
    
}/* adjust */

// the x.nr complex eigenvalues will be stored in an x.nr x 1 matc, where x.nr = eigenvalues.nr
void matcEigenvectors(matc& eigenvectors, matc& eigenvalues, long*& algMultiplicity, long*& geoMultiplicity, const matc& x, long myDecPrec, const fp& zeroSet)
{
    matc            *theV;
    fpComplex       *pArray, e;  // will hold eigenvalues
    matc            xt, nullVectors;
    fpComplex       *eList1;
    matc            eList;
    long            i, j, k, kk;
    int             numNV;
    
    if(x.nr!=x.nc || x.nr<1)
        return;
    
    if(x.nr==1)
    {
        return;
    }
    
    if(!matcEigenvalues(eList1, pArray, x, myDecPrec))
        return;
    
    init(eList, x.nr, 1);
    for(i=0;i<x.nr;++i)
        eList.array[i][0] = eList1[i];
    
    // we must use zeroSet to compare eigenvalues
    adjust(eigenvalues, algMultiplicity, eList, zeroSet);
    geoMultiplicity = (long*)malloc(eigenvalues.nr*sizeof(long));
    
    theV = (matc*)malloc(eigenvalues.nr*sizeof(matc));
    for(k=0;k<eigenvalues.nr;++k)
    {
        e = eigenvalues.array[k][0];
        xt = x;
        for(i=0;i<x.nr;++i)
            xt.array[i][i] = xt.array[i][i] - e;
        matcNullSpace(numNV, nullVectors, xt, zeroSet);
        geoMultiplicity[k] = numNV;
        init(theV[k], x.nr, numNV);
        for(j=0;j<numNV;++j)
            for(i=0;i<x.nr;++i)
                theV[k].array[i][j] = nullVectors.array[i][j];
        myFree(nullVectors);
    }
    
    // load eigenvectors
    numNV = 0;
    kk = 0;
    for(k=0;k<eigenvalues.nr;++k)
    {
        for(j=0;j<theV[k].nc;++j)
            numNV++;
    }
    
    init(eigenvectors, x.nr, numNV);
    for(k=0;k<eigenvalues.nr;++k)
    {
        for(j=0;j<theV[k].nc;++j)
        {
            for(i=0;i<x.nr;++i)
            {
               eigenvectors.array[i][kk] = theV[k].array[i][j];
            }
            kk++;
        }
    }
    
    for(k=0;k<eigenvalues.nr;++k)
        myFree(theV[k]);
    
}/* matcEigenvectors */
