/* RBinterface.cp */

#include "rb_plugin.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "matinv.h"
#include "diags.h"
#include "myMatrixCalc.h"
#include "CalcElement.h"
#include "SVD.h"

typedef struct
{
	double		**array;
	long		nr;	// number of rows
	long		nc;	// number of columns
	
} MatrixData;

#define InvalidNumberExcName	"MatrixBadEntryException"

#define kCFStringEncodingASCII 0x0600
#define kCFStringEncodingUTF8 0x08000100
#define kCFStringEncodingUTF16 0x0100


extern REALclassDefinition		MatrixClass;

static long Min(long x, long y)
{
	if(x<=y)
		return x;
	
	return y;
	
}/* min */


static void MatrixConstructor(REALobject instance)
{
	MatrixData		*data;
	
	// Get the class data
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil)
		return;
	
	data->array = nil;
	data->nr = 0;
	data->nc = 0;
	
}/* MatrixConstructor */


static void NewMatrix(REALobject instance, long numberOfRows, long numberOfColumns)
{
	MatrixData		*data;
	long			i, j;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil || numberOfRows<=0 || numberOfColumns<=0)
		return;
	
	data->nr = numberOfRows;
	data->nc = numberOfColumns;
	data->array = (double **)malloc(data->nr*sizeof(double *));
	for (i=0;i<data->nr;i++)
		(data->array)[i] = (double *)malloc(data->nc*sizeof(double));
	
	for(i=0;i<data->nr;i++)
		for(j=0;j<data->nc;j++)
			(data->array)[i][j] = 0;
	
}/* NewMatrix */



// I'll assume the memory block has the endian of the platform
static void NewMatrixMB(REALobject instance, long numberOfRows, long numberOfColumns, REALmemoryBlock mb, bool byRow)
{
	double				*mbPtr;
	long				mbSize;
	MatrixData			*data;
	long				i, j;
	
	if(mb==nil)
		return;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil || numberOfRows<=0 || numberOfColumns<=0)
		return;
	
	mbSize = REALMemoryBlockGetSize(mb);
	if((8*numberOfRows*numberOfColumns)>mbSize)
		return;
	
	mbPtr = (double*)REALMemoryBlockGetPtr(mb);
	if(mbPtr==nil)
		return;
	
	data->nr = numberOfRows;
	data->nc = numberOfColumns;
	data->array = (double **)malloc(data->nr*sizeof(double *));
	for (i=0;i<data->nr;i++)
		(data->array)[i] = (double *)malloc(data->nc*sizeof(double));
	
	if(byRow)
	{
		for(i=0;i<numberOfRows;i++)
		{
			for(j=0;j<numberOfColumns;j++)
			{
				(data->array)[i][j] = *mbPtr++;
			}
		}
	}
	else
	{
		for(j=0;j<numberOfColumns;j++)
		{
			for(i=0;i<numberOfRows;i++)
			{
				(data->array)[i][j] = *mbPtr++;
			}
		}
	}
	
}/* NewMatrixMB */


static void NewMatrixS(REALobject instance, long numberOfRows, long numberOfColumns, REALstring MStr)
{
	MatrixData		*data;
	long			i, j, k, r, kd, rd, length, rowLength;
	char			*dataString, *rowString, *doubleString, ch, CR=0x0D, SP=0x20, LF=0x0A,TAB=0x09;
	double			theDouble;
	bool			endOfData, endOfRowData, areInNumber;
	
	endOfData = false;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil || numberOfRows<=0 || numberOfColumns<=0)
		return;
	
	data->nr = numberOfRows;
	data->nc = numberOfColumns;
	data->array = (double **)malloc(data->nr*sizeof(double *));
	for (i=0;i<data->nr;i++)
		(data->array)[i] = (double *)malloc(data->nc*sizeof(double));
	
	// now get the matrix elements from MStr
	length = MStr->Length();
	
	// if length==0 make it a null matrix and return
	if(length==0)
	{
		for(i=0;i<data->nr;i++)
			for(j=0;j<data->nc;j++)
				(data->array)[i][j] = 0;
		return;
	}
	
	// length!=0
	dataString = (char *)malloc((length+1)*sizeof(char));
	rowString = (char *)malloc((length+1)*sizeof(char));
	doubleString = (char *)malloc((length+1)*sizeof(char));
	memcpy(dataString, (char*)MStr->CString(), length); // to, from
	dataString[length] = 0; // makes it a C string
	
	/*
	 for example, a 2 x 3 matrix string would look like:
	 1 2 3
	 4 5 6
	 
	 Mac newline = 0xD (CR), Win32 newline = 0xD 0xA (CR LF), Unix newline = 0xA (LF)
	 */
	
	k = 0; // start index of dataString
	for(i=0;i<data->nr;i++)
	{
		if(k>=length)
			endOfData = true;
		
		if(!endOfData)
		{
			r = 0; // start index of rowString
			// get ith row as rowString, read to CR or end of dataString
			ch = dataString[k];
			//while(k<length && ch!=CR)
			while(k<length && ch!=CR && ch!=LF)
			{
				rowString[r] = ch;
				r++;
				k++;
				ch = dataString[k];
			}
			rowString[r] = 0;
			rowLength = strlen(rowString);
			if(rowLength==0)
				endOfRowData = true;
			else
				endOfRowData = false;
			
			while(!(ch=='+' || ch=='-' || ch=='e' || ch=='E' || ch=='.' || (ch>=0x30 && ch<=0x39)))
				ch = dataString[++k];
			
			// now either ch==CR or ch==LF or ch==0 because we've reached the end of dataString
			// skip over it
			//k++;
			
			/*
			 #if MYWIN32
			 k++; // skip over LF
			 #endif
			 */
			
			
			kd = 0; // start index of rowString
		}
		for(j=0;j<data->nc;j++)
		{
			if(!endOfRowData && !endOfData)
			{
				areInNumber = false;
				rd = 0; // start index of doubleString
				// get jth double string as doubleString, read to SP or end of rowString
				ch = rowString[kd];
				
				while(kd<rowLength && ch==SP)
				{
					//doubleString[rd] = ch;
					//rd++;
					kd++;
					ch = rowString[kd]; 
				}
				
				// now into number, go until whitespace
				while(kd<rowLength && !(ch==SP || ch==TAB))
				{
					doubleString[rd] = ch;
					rd++;
					kd++;
					ch = rowString[kd]; 
				}
				
				doubleString[rd] = 0;
				
				if(kd>=rowLength)
					endOfRowData = true;
				
				if(!calcElement(theDouble, doubleString))
				{
					theDouble = 0;
					REALobject err = REALnewInstance(InvalidNumberExcName);
					REALRaiseException(err);
				}
				
				(data->array)[i][j] = theDouble;
			}
			else
				(data->array)[i][j] = 0;
		}
		
	}
	
	free(dataString);
	free(rowString);
	free(doubleString);
	
}/* NewMatrixS */


/* -------------------------------------
 MatrixDestructor
 
 Called by REALbasic when a Matrix object is destroyed.
 
 instance = the Matrix object being destroyed
 ------------------------------------- */

static void MatrixDestructor(REALobject instance)
{
	MatrixData		*data;
	long			i;
	
	if(instance==nil)
		return;
	
	// Get the class data
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil || data->array==nil)
		return;
	
	for (i=0;i<data->nr;i++)
		free((data->array)[i]);
	
	free(data->array);
	
}/* MatrixDestructor */


// access functions


static REALstring MatrixOut(REALobject instance)
{
	MatrixData		*data;
	char			*myString, ***doubleStringMatrix, CR=0x0D, SP=0x20, LF=0x0A;
	long			i, j, k, length, len, diffLen;
	
	REALstring nullString = REALBuildString (nil, 0);
	
	if(instance==nil)
		return nullString;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	if(data==nil)
		return nullString;
	
	length = 20; // length of string double should be less than this
	
	doubleStringMatrix = (char ***)malloc(data->nr*sizeof(char *));
	for (i=0;i<data->nr;i++)
		doubleStringMatrix[i] = (char **)malloc(data->nc*sizeof(char*));
	
	for(i=0;i<data->nr;i++)
		for(j=0;j<data->nc;j++)
			doubleStringMatrix[i][j] = (char *)malloc(length);
	
	
	// put doubles into doubleStringMatrix
	// if it's positive put one space at start, so negatives will align with positives in a column
	for(i=0;i<data->nr;i++)
		for(j=0;j<data->nc;j++)
			if(j<(data->nc-1))
			{
				if(data->array[i][j]>=0)
					sprintf(doubleStringMatrix[i][j]," %.7g  ", data->array[i][j]);
				else
					sprintf(doubleStringMatrix[i][j],"%.7g  ", data->array[i][j]);
			}
			else
			{
				if(data->array[i][j]>=0)
					sprintf(doubleStringMatrix[i][j]," %.7g", data->array[i][j]);
				else
					sprintf(doubleStringMatrix[i][j],"%.7g", data->array[i][j]);
			}
	
	
	// find the largest strlen
	length = 0;
	for(i=0;i<data->nr;i++)
		for(j=0;j<data->nc;j++)
		{
			len = strlen(doubleStringMatrix[i][j]);
			if(len>length)
				length = len;
		}
	
	//length++;
	
	// now pad each double string on right so it has length as its strlen, except for the last column
	
	for(i=0;i<data->nr;i++)
		for(j=0;j<(data->nc-1);j++)
		{
			len = strlen(doubleStringMatrix[i][j]);
			diffLen = length - len;
			for(k=0;k<diffLen;k++)
				doubleStringMatrix[i][j][len+k] = SP;
			
			doubleStringMatrix[i][j][length] = 0;
		}
	
	length = 20*data->nr*data->nc + data->nr + 1; // estimate of maximum length of C string returned
	
	myString = (char *)malloc(length*sizeof(char));
	myString[0] = 0;
	
	// load up myString
	for(i=0;i<data->nr;i++)
	{
		for(j=0;j<data->nc;j++)
			strcat(myString, doubleStringMatrix[i][j]);
		
		len = strlen(myString);
		myString[len] = CR;
		
#if MYWIN32
		myString[++len] = LF;
#endif
		myString[++len] = 0;
	}
	
	length = strlen(myString);
	
	REALstring z=REALBuildString (nil, length);
	memcpy((char*)z->CString(), myString, length); // to, from
	
	free(myString);
	
	for(i=0;i<data->nr;i++)
		for(j=0;j<data->nc;j++)
			free(doubleStringMatrix[i][j]);
	
	for(i=0;i<data->nr;i++)
		free(doubleStringMatrix[i]);
	
	free(doubleStringMatrix);
	
	REALSetStringEncoding(z, kCFStringEncodingUTF8);
	
	return z;
	
}/* MatrixOut */


// MatrixOut2 outputs as "1, 2;3, 4"
static REALstring MatrixOut2(REALobject instance)
{
	MatrixData		*data;
	char			*myString, doubleString[20];
	long			i, j, length;
	
	REALstring nullString = REALBuildString (nil, 0);
	
	if(instance==nil)
		return nullString;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	if(data==nil)
		return nullString;
	
	length = 20*data->nr*data->nc + data->nr + 1; // estimate of maximum length of C string returned
	
	myString = (char *)malloc(length*sizeof(char));
	myString[0] = 0;
	
	for(i=0;i<data->nr;i++)
	{
		for(j=0;j<data->nc;j++)
		{
			if(j<(data->nc-1))
				sprintf(doubleString, "%.7g, ", (data->array)[i][j]);
			else
				sprintf(doubleString, "%.7g", (data->array)[i][j]);
			
			strcat(myString, doubleString);
		}
		
		if(i<(data->nr-1))
		{
			length = strlen(myString);
			myString[length] = ' ';
			myString[++length] = ';';
			myString[++length] = ' ';
			myString[++length] = 0;
		}
	}
	
	length = strlen(myString);
	
	REALstring z = REALBuildString (nil, length);
	memcpy((char*)z->CString(), myString, length); // to, from
	
	free(myString);
	
	REALSetStringEncoding(z, kCFStringEncodingUTF8);
	
	return z;
	
}/* MatrixOut2 */


static double Element(REALobject instance, long i, long j)
{
	MatrixData		*data;
	
	if(instance==nil)
		return NAN;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil || i>=data->nr || j>=data->nc || i<0 || j<0)
		return NAN;
	
	return (data->array)[i][j];
	
}/* Element */


static REALobject MatrixNegate(REALobject instance)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(instance==nil)
		return instance;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	
	zData->array = (double **)malloc(data->nr*sizeof(double *));
	for (i=0;i<data->nr;i++)
		(zData->array)[i] = (double *)malloc(data->nc*sizeof(double));
	
	for(i=0;i<data->nr;i++)
		for(j=0;j<data->nc;j++)
			(zData->array)[i][j] = -(data->array)[i][j];
	
	return z;
	
}/* MatrixNegate */


static REALobject AddMM(REALobject instance, REALobject x)
{
	MatrixData		*data, *xData, *zData;
	REALobject		z;
	long			i, j;
	
	if(instance==nil)
		return instance;
	
	if(x==nil)
		return x;
	
	xData = (MatrixData *) REALGetClassData(x, &MatrixClass);
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil || xData==nil || xData->nr!=data->nr || xData->nc!=data->nc)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = xData->nr;
	zData->nc = xData->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = (data->array)[i][j] + (xData->array)[i][j];
	
	return z;
	
}/* AddMM */


static REALobject AddMD(REALobject instance, double x)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(instance==nil)
		return instance;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			if(i!=j)
				(zData->array)[i][j] = (data->array)[i][j];
			else
				(zData->array)[i][i] = (data->array)[i][i] + x;
	
	return z;
	
}/* AddMD */


static REALobject SubMM(REALobject instance, REALobject x)
{
	MatrixData		*data, *xData, *zData;
	REALobject		z;
	long			i, j;
	
	if(instance==nil)
		return instance;
	
	if(x==nil)
		return x;
	
	xData = (MatrixData *) REALGetClassData(x, &MatrixClass);
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil || xData==nil || xData->nr!=data->nr || xData->nc!=data->nc)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = xData->nr;
	zData->nc = xData->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = (data->array)[i][j] - (xData->array)[i][j];
	
	return z;
	
}/* SubMM */


static REALobject SubMD(REALobject instance, double x)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(instance==nil)
		return instance;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			if(i!=j)
				(zData->array)[i][j] = (data->array)[i][j];
			else
				(zData->array)[i][i] = (data->array)[i][i] - x;
	
	return z;
	
}/* SubMD */


static REALobject SubDM(REALobject instance, double x)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(instance==nil)
		return instance;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			if(i!=j)
				(zData->array)[i][j] = -(data->array)[i][j];
			else
				(zData->array)[i][i] = -(data->array)[i][i] + x;
	
	return z;
	
}/* SubDM */


static REALobject MulMM(REALobject instance, REALobject x)
{
	MatrixData		*data, *xData, *zData;
	REALobject		z;
	long			i, j, k;
	
	if(instance==nil)
		return instance;
	
	if(x==nil)
		return x;
	
	xData = (MatrixData *) REALGetClassData(x, &MatrixClass);
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil || xData==nil || data->nc!=xData->nr)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = data->nr;
	zData->nc = xData->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
		{
			(zData->array)[i][j] = 0;
			for(k=0;k<data->nc;k++)
				(zData->array)[i][j] = (zData->array)[i][j] + (data->array)[i][k] * (xData->array)[k][j];
		}
	
	return z;
	
}/* MulMM */


static REALobject MulDM(REALobject instance, double x)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(instance==nil)
		return instance;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = x*(data->array)[i][j];
	
	return z;
	
}/* MulDM */


static REALobject DivMD(REALobject instance, double x)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(instance==nil)
		return instance;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			if(x!=0)
				(zData->array)[i][j] = (data->array)[i][j]/x;
			else
				(zData->array)[i][j] = NAN;
	
	return z;
	
}/* DivMD */


static REALobject DivDM(REALobject instance, double x)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	double			determinant;
	
	if(instance==nil)
		return instance;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = (data->array)[i][j];
	
	determinant = matinv(zData->array, zData->nr);
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			if(determinant!=0)
				(zData->array)[i][j] = x*(zData->array)[i][j];
			else
				(zData->array)[i][j] = NAN;
	
	return z;
	
}/* DivDM */


static REALobject DivMM(REALobject instance, REALobject x)
{
	MatrixData		*data, *xData, *zData;
	REALobject		z;
	long			i, j, k;
	double			**array, determinant;
	
	if(instance==nil)
		return instance;
	
	if(x==nil)
		return x;
	
	xData = (MatrixData *) REALGetClassData(x, &MatrixClass);
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil || xData==nil || xData->nc!=xData->nr || data->nc!=xData->nr)
		return instance;
	
	array = (double **)malloc(xData->nr*sizeof(double *));
	for (i=0;i<xData->nr;i++)
		array[i] = (double *)malloc(xData->nc*sizeof(double));
	
	for(i=0;i<xData->nr;i++)
		for(j=0;j<xData->nc;j++)
			array[i][j] = (xData->array)[i][j];
	
	determinant = matinv(array, xData->nr);
	
	if(determinant==0)
		return instance;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	zData->nr = data->nr;
	zData->nc = xData->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
		{
			(zData->array)[i][j] = 0;
			for(k=0;k<xData->nc;k++)
				(zData->array)[i][j] = (zData->array)[i][j] + (data->array)[i][k] * array[k][j];
		}
	
	for (i=0;i<xData->nr;i++)
		free(array[i]);
	
	free(array);
	
	return z;
	
}/* DivMM */


static int CompareMM(REALobject instance, REALobject x)
{
	MatrixData		*data, *xData;
	long			i, j;
	
	if(instance==nil)
		return 0;
	
	if(x==nil)
		return 0;
	
	xData = (MatrixData *) REALGetClassData(x, &MatrixClass);
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if(data==nil || xData==nil)
		return 0;
	
	if(data->nr!=xData->nr || data->nc!=xData->nc)
		return 1;
	
	for(i=0;i<data->nr;i++)
		for(j=0;j<data->nc;j++)
		{
			if((data->array)[i][j] < (xData->array)[i][j])
				return -1;
			else
				if((data->array)[i][j] > (xData->array)[i][j])
					return 1;
		}
	
	
	return 0;
	
}/* CompareMM */


static long NumRows(REALobject instance)
{
	MatrixData		*data;
	
	if(instance==nil)
		return 0;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil)
		return 0;
	
	return data->nr;
	
}/* NumRows */


static long NumColumns(REALobject instance)
{
	MatrixData		*data;
	
	if(instance==nil)
		return 0;
	
	data = (MatrixData *) REALGetClassData(instance, &MatrixClass);
	
	if (data == nil)
		return 0;
	
	return data->nc;
	
}/* NumColumns */


// REALbasic class & method definitions for the Matrix object

static REALmethodDefinition MatrixMethods[] = {
	// defines methods:
	{ (REALproc)MatrixConstructor, REALnoImplementation, "Constructor()" },
	{ (REALproc)NewMatrix, REALnoImplementation, "Constructor(numberOfRows As Integer, numberOfColumns As Integer)" },
	{ (REALproc)NewMatrixMB, REALnoImplementation, "Constructor(numberOfRows As Integer, numberOfColumns As Integer, mb As MemoryBlock, byRow As Boolean)" },
	{ (REALproc)NewMatrixS, REALnoImplementation, "Constructor(numberOfRows As Integer, numberOfColumns As Integer, matrixString As String)" },
	{ (REALproc)MatrixOut, REALnoImplementation, "Str() As String" },
	{ (REALproc)MatrixOut2, REALnoImplementation, "Str2() As String" },
	{ (REALproc)Element, REALnoImplementation, "Element(rowIndex As Integer, columnIndex As Integer) As Double" },
	{ (REALproc)MatrixNegate, REALnoImplementation, "Operator_Negate() As Matrix" },
	{ (REALproc)AddMM, REALnoImplementation, "Operator_Add(x As Matrix) As Matrix" },
	{ (REALproc)AddMD, REALnoImplementation, "Operator_Add(x As Double) As Matrix" },
	{ (REALproc)AddMD, REALnoImplementation, "Operator_AddRight(x As Double) As Matrix" },
	{ (REALproc)SubMM, REALnoImplementation, "Operator_Subtract(x As Matrix) As Matrix" },
	{ (REALproc)SubMD, REALnoImplementation, "Operator_Subtract(x As Double) As Matrix" },
	{ (REALproc)SubDM, REALnoImplementation, "Operator_SubtractRight(x As Double) As Matrix" },
	{ (REALproc)MulMM, REALnoImplementation, "Operator_Multiply(x As Matrix) As Matrix" },
	{ (REALproc)MulDM, REALnoImplementation, "Operator_Multiply(x As Double) As Matrix" },
	{ (REALproc)MulDM, REALnoImplementation, "Operator_MultiplyRight(x As Double) As Matrix" },
	{ (REALproc)DivMD, REALnoImplementation, "Operator_Divide(x As Double) As Matrix" },
	{ (REALproc)DivDM, REALnoImplementation, "Operator_DivideRight(x As Double) As Matrix" },
	{ (REALproc)DivMM, REALnoImplementation, "Operator_Divide(x As Matrix) As Matrix" },
	{ (REALproc)CompareMM, REALnoImplementation, "Operator_Compare(x As Matrix) As Integer" },
	{ (REALproc)NumRows, REALnoImplementation, "NumRows() As Integer" },
	{ (REALproc)NumColumns, REALnoImplementation, "NumColumns() As Integer" },
};


REALclassDefinition MatrixClass = {
	kCurrentREALControlVersion,
	"Matrix",                    			// name of class
	nil,                                    // no superclasses
	sizeof(MatrixData),                      // size of our data
	0,																			// --reserved--
	(REALproc) MatrixConstructor,     				// constructor
	(REALproc) MatrixDestructor,							// destructor
	nil,            												// properties
	0,																			// property count
	MatrixMethods,               					// methods
	sizeof(MatrixMethods) / sizeof(REALmethodDefinition),	// method count
	// other fields left nil -- no events etc.
};

// the following pragma is essential here

#pragma mark -

/*
 static REALobject NewMatrix(long numberOfRows, long numberOfColumns)
 {
 MatrixData		*zData;
 REALobject		z;
 long			i, j;
 
 z = REALnewInstance("Matrix");
 zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
 
 zData->nr = numberOfRows;
 zData->nc = numberOfColumns;
 zData->array = (double **)malloc(zData->nr*sizeof(double *));
 for (i=0;i<zData->nr;i++)
 (zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
 
 for(i=0;i<zData->nr;i++)
 for(j=0;j<zData->nc;j++)
 (zData->array)[i][j] = 0;
 
 return z;
 
 }*//* NewMatrix */


// I'll assume the memory block has the endian of the platform
static REALobject CreateMatrixMB(long numberOfRows, long numberOfColumns, REALmemoryBlock mb, bool byRow)
{
	double				*mbPtr;
	long				mbSize;
	REALobject			z;
	MatrixData			*zData;
	long				i, j;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	if(mb==nil)
		return z;
	
	if (zData == nil || numberOfRows<=0 || numberOfColumns<=0)
		return z;
	
	mbSize = REALMemoryBlockGetSize(mb);
	if((8*numberOfRows*numberOfColumns)>mbSize)
		return z;
	
	mbPtr = (double*)REALMemoryBlockGetPtr(mb);
	if(mbPtr==nil)
		return z;
	
	zData->nr = numberOfRows;
	zData->nc = numberOfColumns;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	if(byRow)
	{
		for(i=0;i<numberOfRows;i++)
		{
			for(j=0;j<numberOfColumns;j++)
			{
				(zData->array)[i][j] = *mbPtr++;
			}
		}
	}
	else
	{
		for(j=0;j<numberOfColumns;j++)
		{
			for(i=0;i<numberOfRows;i++)
			{
				(zData->array)[i][j] = *mbPtr++;
			}
		}
	}
	
	return z;
	
}/* NewMatrixMB */


/* uses input format "1,2,3;4,5,6" */
static REALobject CreateMatrixS(long numberOfRows, long numberOfColumns, REALstring MStr)
{
	MatrixData		*zData;
	long			i, j, k, r, kd, rd, length, rowLength;
	char			*dataString, *rowString, *doubleString, ch;
	double			theDouble;
	bool			endOfData, endOfRowData;
	REALobject		z;
	
	endOfData = false;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	if (zData == nil || numberOfRows<=0 || numberOfColumns<=0)
		return z;
	
	zData->nr = numberOfRows;
	zData->nc = numberOfColumns;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	// now get the matrix elements from MStr
	length = MStr->Length();
	
	// if length==0 make it a null matrix and return
	if(length==0)
	{
		for(i=0;i<zData->nr;i++)
			for(j=0;j<zData->nc;j++)
				(zData->array)[i][j] = 0;
		return z;
	}
	
	// length!=0
	dataString = (char *)malloc((length+1)*sizeof(char));
	rowString = (char *)malloc((length+1)*sizeof(char));
	doubleString = (char *)malloc((length+1)*sizeof(char));
	memcpy(dataString, (char*)MStr->CString(), length); // to, from
	dataString[length] = 0; // makes it a C string
	
	// for example, a 2 x 3 matrix string would look like: "1,2,3;4,5,6"
	k = 0; // start index of dataString
	for(i=0;i<zData->nr;i++)
	{
		if(k>=length)
			endOfData = true;
		
		if(!endOfData)
		{
			r = 0; // start index of rowString
			// get ith row as rowString, read to ; or end of dataString
			ch = dataString[k];
			while(k<length && ch!=';')
			{
				rowString[r] = ch;
				r++;
				k++;
				ch = dataString[k];
			}
			rowString[r] = 0;
			rowLength = strlen(rowString);
			if(rowLength==0)
				endOfRowData = true;
			else
				endOfRowData = false;
			
			// now either ch==';' or ch==0 because we've reached the end of dataString
			k++; // skip over ";" if necessary
			
			kd = 0; // start index of rowString
		}
		for(j=0;j<zData->nc;j++)
		{
			if(!endOfRowData && !endOfData)
			{
				rd = 0; // start index of doubleString
				// get jth double string as doubleString, read to , or end of rowString
				ch = rowString[kd];
				while(kd<rowLength && ch!=',')
				{
					doubleString[rd] = ch;
					rd++;
					kd++;
					ch = rowString[kd]; 
				}
				
				doubleString[rd] = 0;
				
				if(kd>=rowLength)
					endOfRowData = true;
				else
					kd++; // skip over ","
				
				if(!calcElement(theDouble, doubleString))
				{
					theDouble = 0;
					REALobject err = REALnewInstance(InvalidNumberExcName);
					REALRaiseException(err);
				}
				
				(zData->array)[i][j] = theDouble;
			}
			else
				(zData->array)[i][j] = 0;
		}
		
	}
	
	free(dataString);
	free(rowString);
	free(doubleString);
	
	return z;
	
}/* CreateMatrixS */



// this makes Matrix object mutable
static REALobject ChangeElement(REALobject theMatrix, long rowIndex, long columnIndex, double value)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || rowIndex>=data->nr || columnIndex>=data->nc || rowIndex<0 || columnIndex<0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	if(zData==nil)
		return theMatrix;
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = (data->array)[i][j];
	
	(zData->array)[rowIndex][columnIndex] = value;
	
	return z;
	
}/* ChangeElement */


static void NewElement(REALobject theMatrix, long rowIndex, long columnIndex, double value)
{
	MatrixData		*data;
	
	if(theMatrix==nil)
		return;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr<1 || data->nc<1 || rowIndex>=data->nr || columnIndex>=data->nc || rowIndex<0 || columnIndex<0)
	{
		REALobject err = REALnewInstance("RowColumnMismatch");
		REALRaiseException(err);
		return;
	}
	
	(data->array)[rowIndex][columnIndex] = value;
	
}/* NewElement */


static REALobject Inverse(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	double			determinant;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = (data->array)[i][j];
	
	determinant = matinv(zData->array, zData->nr);
	
	if(determinant==0)
	{
		for(i=0;i<zData->nr;i++)
			for(j=0;j<zData->nc;j++)
				(zData->array)[i][j] = 0;
	}
	
	return z;
	
}/* Inverse */


static REALobject InverseDet(REALobject theMatrix, double& determinant)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = (data->array)[i][j];
	
	determinant = matinv(zData->array, zData->nr);
	
	if(determinant==0)
	{
		for(i=0;i<zData->nr;i++)
			for(j=0;j<zData->nc;j++)
				(zData->array)[i][j] = 0;
	}
	
	return z;
	
}/* InverseDet */


static double Determinant(REALobject theMatrix)
{
	MatrixData		*data;
	long			i, j;
	double			**array, determinant;
	
	if(theMatrix==nil)
		return NAN;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return NAN;
	
	array = (double **)malloc(data->nr*sizeof(double *));
	for (i=0;i<data->nr;i++)
		array[i] = (double *)malloc(data->nc*sizeof(double));
	
	for(i=0;i<data->nr;i++)
		for(j=0;j<data->nc;j++)
			array[i][j] = (data->array)[i][j];
	
	determinant = matinv(array, data->nr);
	
	for (i=0;i<data->nr;i++)
		free(array[i]);
	
	free(array);
	
	return determinant;
	
}/* Determinant */


static REALobject Transpose(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i, j;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nc;
	zData->nc = data->nr;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = (data->array)[j][i];
	
	return z;
	
}/* Transpose */


static REALobject Diagonalize(REALobject theMatrix, REALobject EigenvectorMatrix, double eps)
{
	MatrixData		*data, *zData, *EData;
	REALobject		z;
	long			i,j;
	
	if(theMatrix==nil)
		return theMatrix;
	
	if(EigenvectorMatrix==nil)
		return EigenvectorMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	EData = (MatrixData *) REALGetClassData(EigenvectorMatrix, &MatrixClass);
	
	if (EData==nil || data == nil || data->nr!=data->nc || data->nr<=1 || EData->nr!=data->nr ||  EData->nc!=data->nc || eps<=0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	for(i=0;i<zData->nr;i++)
		for(j=0;j<zData->nc;j++)
			(zData->array)[i][j] = (data->array)[i][j];
	
	/*
	 EigenvectorMatrix = REALnewInstance("Matrix");
	 EData = (MatrixData *) REALGetClassData(EigenvectorMatrix, &MatrixClass);
	 EData->nr = data->nr;
	 EData->nc = data->nc;
	 EData->array = (double **)malloc(EData->nr*sizeof(double *));
	 for(i=0;i<EData->nr;i++)
	 (EData->array)[i] = (double *)malloc(EData->nc*sizeof(double));
	 */
	
	diags(zData->array, EData->array, zData->nr, eps);
	
	// make z matrix symmetric
	for(i=0;i<zData->nr;i++)
		for(j=i+1;j<zData->nc;j++)
			(zData->array)[j][i] = (zData->array)[i][j];
	
	return z;
	
}/* Diagonalize */


// returns U, M, V, and D must have been initialized
static REALobject mySVD(REALobject M, REALobject V, REALobject D, double eps)
{
	MatrixData		*MData, *VData, *DData, *UData;
	REALobject		U;
	double			*d;
	long			m, n, i, j;
	
	if(M==nil || V==nil || D==nil || eps<=0)
		return M;
	
	MData = (MatrixData *) REALGetClassData(M, &MatrixClass);
	VData = (MatrixData *) REALGetClassData(V, &MatrixClass);
	DData = (MatrixData *) REALGetClassData(D, &MatrixClass);
	
	
	if(MData==nil ||VData==nil || DData==nil)
		return M;
	
	m = MData->nr;
	n = MData->nc;
	
	if(m<n || VData->nr!=n || VData->nc!=n || DData->nr!=n || DData->nc!=n)
	{
		REALobject err = REALnewInstance("MatrixBadEntryException");
		REALRaiseException(err);
		return M;
	}
	
	U = REALnewInstance("Matrix");
	UData = (MatrixData *) REALGetClassData(U, &MatrixClass);
	
	UData->nr = m;
	UData->nc = n;
	UData->array = (double **)malloc(m*sizeof(double *));
	for (i=0;i<m;i++)
		(UData->array)[i] = (double *)malloc(n*sizeof(double));
	
	d = (double *)malloc(n*sizeof(double));
	
	svd(UData->array, d, VData->array, MData->array, m, n, eps);
	
	for(i=0;i<n;i++)
		for(j=0;j<n;j++)
		{
			if(i==j)
				DData->array[i][j] = d[i];
			else
				DData->array[i][j] = 0;
		}
	
	free(d);
	
	return U;
	
}/* mySVD */


// returns U, M, V, and D must have been initialized
static REALobject mySVD2(REALobject M, REALobject V, REALobject D, double eps)
{
	MatrixData		*MData, *VData, *DData, *UData;
	REALobject		U;
	double			*d;
	long			m, n, i, j;
	
	if(M==nil || V==nil || D==nil || eps<=0)
		return M;
	
	MData = (MatrixData *) REALGetClassData(M, &MatrixClass);
	VData = (MatrixData *) REALGetClassData(V, &MatrixClass);
	DData = (MatrixData *) REALGetClassData(D, &MatrixClass);
	
	
	if(MData==nil ||VData==nil || DData==nil)
		return M;
	
	m = MData->nr;
	n = MData->nc;
	
	if(m<n || VData->nr!=n || VData->nc!=n || DData->nr!=m || DData->nc!=n)
	{
		REALobject err = REALnewInstance("MatrixBadEntryException");
		REALRaiseException(err);
		return M;
	}
	
	U = REALnewInstance("Matrix");
	UData = (MatrixData *) REALGetClassData(U, &MatrixClass);
	
	UData->nr = m;
	UData->nc = m;
	UData->array = (double **)malloc(m*sizeof(double *));
	for (i=0;i<m;i++)
		(UData->array)[i] = (double *)malloc(m*sizeof(double));
	
	d = (double *)malloc(n*sizeof(double));
	
	svd2(UData->array, d, VData->array, MData->array, m, n, eps);
	
	// fill upper duagonal matrix with sindular values
	for(i=0;i<n;i++)
	{
		for(j=0;j<n;j++)
		{
			if(i==j)
				DData->array[i][j] = d[i];
			else
				DData->array[i][j] = 0;
		}
	}
	// fill bottom null matrix
	for(i=n;i<m;i++)
		for(j=0;j<n;j++)
			DData->array[i][j] = 0;
	
	free(d);
	
	return U;
	
}/* mySVD2 */


static REALobject SqrM(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	mySqr(zData->array, data->array, zData->nr);
	
	return z;
	
}/* SqrM */


static REALobject ExpM(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	myExp(zData->array, data->array, zData->nr);
	
	return z;
	
}/* ExpM */


static REALobject SinM(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	mySin(zData->array, data->array, zData->nr);
	
	return z;
	
}/* SinM */


static REALobject CosM(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	myCos(zData->array, data->array, zData->nr);
	
	return z;
	
}/* CosM */


static REALobject TanM(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	myTan(zData->array, data->array, zData->nr);
	
	return z;
	
}/* TanM */


static REALobject SinhM(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	mySinh(zData->array, data->array, zData->nr);
	
	return z;
	
}/* SinhM */


static REALobject CoshM(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	myCosh(zData->array, data->array, zData->nr);
	
	return z;
	
}/* CoshM */


static REALobject TanhM(REALobject theMatrix)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	myTanh(zData->array, data->array, zData->nr);
	
	return z;
	
}/* TanhM */


static REALobject Commutator(REALobject x, REALobject y)
{
	MatrixData		*xData, *yData, *zData;
	REALobject		z;
	long			i;
	
	if(x==nil)
		return x;
	
	if(y==nil)
		return y;
	
	xData = (MatrixData *) REALGetClassData(x, &MatrixClass);
	yData = (MatrixData *) REALGetClassData(y, &MatrixClass);
	
	if(xData==nil || yData==nil || xData->nr!=xData->nc || yData->nr!=yData->nc || xData->nr!=yData->nr || xData->nr<1)
		return x;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = xData->nr;
	zData->nc = xData->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	myCommutator(zData->array, xData->array, yData->array, zData->nr);
	
	return z;
	
}/* Commutator */


static REALmemoryBlock ConvToMB(REALobject matrix, bool byRow)
{
	REALmemoryBlock			mb;
	double					*mbPtr;
	MatrixData				*data;
	long					i, j;
	
	data = (MatrixData *) REALGetClassData(matrix, &MatrixClass);
	if(data==nil)
		return NULL;
	
	mb = REALNewMemoryBlock(8*data->nr*data->nc);
	mbPtr = (double*)REALMemoryBlockGetPtr(mb);
	if(mbPtr==nil)
		return NULL;
	
	if(byRow)
	{
		for(i=0;i<data->nr;i++)
		{
			for(j=0;j<data->nc;j++)
			{
				*mbPtr++ = (data->array)[i][j];
			}
		}
	}
	else
	{
		for(j=0;j<data->nc;j++)
		{
			for(i=0;i<data->nr;i++)
			{
				*mbPtr++ = (data->array)[i][j];
			}
		}
	}
	
	return mb;
	
}/* ConvToMB */


// numberOfRows and numberOfColumns override nr and nc of matrix
static REALmemoryBlock ConvToMB2(REALobject matrix, long numberOfRows, long numberOfColumns, bool byRow)
{
	REALmemoryBlock			mb;
	double					*mbPtr;
	MatrixData				*data;
	long					i, j;
	
	data = (MatrixData *) REALGetClassData(matrix, &MatrixClass);
	if(data==nil)
		return NULL;
	
	if(numberOfRows<1 || numberOfColumns<1)
		return NULL;
	
	mb = REALNewMemoryBlock(8*numberOfRows*numberOfColumns);
	mbPtr = (double*)REALMemoryBlockGetPtr(mb);
	if(mbPtr==nil)
		return NULL;
	
	if(byRow)
	{
		for(i=0;i<Min(data->nr, numberOfRows);i++)
		{
			for(j=0;j<Min(data->nc, numberOfColumns);j++)
			{
				*mbPtr++ = (data->array)[i][j];
			}
			
			for(j=Min(data->nc, numberOfColumns);j<numberOfColumns;j++)
			{
				*mbPtr++ = 0;
			}
		}
		
		for(i=Min(data->nr, numberOfRows);i<numberOfRows;i++)
		{
			for(j=0;j<Min(data->nc, numberOfColumns);j++)
			{
				*mbPtr++ = 0;
			}
			
			for(j=Min(data->nc, numberOfColumns);j<numberOfColumns;j++)
			{
				*mbPtr++ = 0;
			}
		}
	}
	else
	{
		for(j=0;j<Min(data->nc, numberOfColumns);j++)
		{
			for(i=0;i<Min(data->nr, numberOfRows);i++)
			{
				*mbPtr++ = (data->array)[i][j];
			}
			
			for(i=Min(data->nr, numberOfRows);i<numberOfRows;i++)
			{
				*mbPtr++ = 0;
			}
		}
		
		for(j=Min(data->nc, numberOfColumns);j<numberOfColumns;j++)
		{
			for(i=0;i<Min(data->nr, numberOfRows);i++)
			{
				*mbPtr++ = 0;
			}
			
			for(i=Min(data->nr, numberOfRows);i<numberOfRows;i++)
			{
				*mbPtr++ = 0;
			}
		}
	}
	
	return mb;
	
}/* ConvToMB2 */


static REALobject PowerM(REALobject theMatrix, long n)
{
	MatrixData		*data, *zData;
	REALobject		z;
	long			i;
	long			det;
	
	if(theMatrix==nil)
		return theMatrix;
	
	data = (MatrixData *) REALGetClassData(theMatrix, &MatrixClass);
	
	if (data == nil || data->nr!=data->nc || data->nr==0 || data->nc==0)
		return theMatrix;
	
	z = REALnewInstance("Matrix");
	zData = (MatrixData *) REALGetClassData(z, &MatrixClass);
	
	zData->nr = data->nr;
	zData->nc = data->nc;
	zData->array = (double **)malloc(zData->nr*sizeof(double *));
	for (i=0;i<zData->nr;i++)
		(zData->array)[i] = (double *)malloc(zData->nc*sizeof(double));
	
	det = myPower(zData->array, data->array, n, zData->nr);
	if(!det)
	{
		REALobject err = REALnewInstance("MatrixBadEntryException");
		REALRaiseException(err);
	}
	
	return z;
	
}/* PowerM */



static REALmethodDefinition otherMatrixMethods[] = {
	{ (REALproc)CreateMatrixMB, REALnoImplementation, "CreateMatrix(numberOfRows As Integer, numberOfColumns As Integer, mb As MemoryBlock, byRow As Boolean) As Matrix" },
	{ (REALproc)CreateMatrixS, REALnoImplementation, "CreateMatrix(numberOfRows As Integer, numberOfColumns As Integer, inputString As String) As Matrix" },
	{ (REALproc)ChangeElement, REALnoImplementation, "ChangeElement(theMatrix As Matrix, rowIndex As Integer, columnIndex As Integer, value As Double) As Matrix" },
	{ (REALproc) NewElement, REALnoImplementation, "NewElement(theMatrix As Matrix, rowIndex As Integer, columnIndex As Integer, value As Double)" },
	{ (REALproc)Inverse, REALnoImplementation, "Inverse(theMatrix As Matrix) As Matrix" },
	{ (REALproc)InverseDet, REALnoImplementation, "Inverse(theMatrix As Matrix, ByRef theDeterminant As Double) As Matrix" },
	{ (REALproc)Determinant, REALnoImplementation, "Determinant(theMatrix As Matrix) As Double" },
	{ (REALproc)Transpose, REALnoImplementation, "Transpose(theMatrix As Matrix) As Matrix" },
	{ (REALproc)Diagonalize, REALnoImplementation, "Diagonalize(theMatrix As Matrix, EigenvectorMatrix As Matrix, eps As Double) As Matrix" },
	{ (REALproc)mySVD, REALnoImplementation, "SVD(M As Matrix, V As Matrix, D As Matrix, eps As Double) As Matrix" },
	{ (REALproc)mySVD2, REALnoImplementation, "SVD2(M As Matrix, V As Matrix, D As Matrix, eps As Double) As Matrix" },
	{ (REALproc)SqrM, REALnoImplementation, "Sqr(theMatrix As Matrix) As Matrix" },
	{ (REALproc)ExpM, REALnoImplementation, "Exp(theMatrix As Matrix) As Matrix" },
	{ (REALproc)SinM, REALnoImplementation, "Sin(theMatrix As Matrix) As Matrix" },
	{ (REALproc)CosM, REALnoImplementation, "Cos(theMatrix As Matrix) As Matrix" },
	{ (REALproc)TanM, REALnoImplementation, "Tan(theMatrix As Matrix) As Matrix" },
	{ (REALproc)SinhM, REALnoImplementation, "Sinh(theMatrix As Matrix) As Matrix" },
	{ (REALproc)CoshM, REALnoImplementation, "Cosh(theMatrix As Matrix) As Matrix" },
	{ (REALproc)TanhM, REALnoImplementation, "Tanh(theMatrix As Matrix) As Matrix" },
	{ (REALproc)Commutator, REALnoImplementation, "Commutator(x As Matrix, y As Matrix) As Matrix" },
	{ (REALproc)ConvToMB, REALnoImplementation, "ConvToMemoryBlock(x As Matrix, byRow As Boolean) As MemoryBlock" },
	{ (REALproc)ConvToMB2, REALnoImplementation, "ConvToMemoryBlock(x As Matrix, numberOfRows As Integer, numberOfColumns As Integer, byRow As Boolean) As MemoryBlock" },
	{ (REALproc)PowerM, REALnoImplementation, "Power(theMatrix As Matrix, n As Integer) As Matrix" },
	{ (REALproc)NumRows, REALnoImplementation, "NumRows(theMatrix As Matrix) As Integer" },
	{ (REALproc)NumColumns, REALnoImplementation, "NumColumns(theMatrix As Matrix) As Integer" },
};

REALclassDefinition BadEntryNumberExceptionClass = {
	kCurrentREALControlVersion,
	InvalidNumberExcName,                    			// name of class
	"RuntimeException",
	// other fields left nil -- no events etc.
};


void PluginEntry(void)
{
	int		i;
	
	REALRegisterClass(&MatrixClass);
	
	REALRegisterClass(&BadEntryNumberExceptionClass);
	
	for (i = 0; i < sizeof(otherMatrixMethods) / sizeof(REALmethodDefinition); ++i)
		REALRegisterMethod(&otherMatrixMethods[i]);
}
